package com.atlassian.audit.ao.dao;

import com.atlassian.audit.entity.AuditAttribute;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.ObjectMapper;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static java.util.Objects.requireNonNull;

public class JacksonAttributesSerializer implements AttributesSerializer {
    private final ObjectMapper objectMapper;

    public static class AuditAttributeData {
        private final String name;
        private final String nameI18nKey;
        private final String value;

        @JsonCreator
        public AuditAttributeData(@JsonProperty("nameI18nKey") String nameI18nKey, @JsonProperty("name") String name, @JsonProperty("value") String value) {
            this.nameI18nKey = nameI18nKey == null ? name : nameI18nKey;
            this.name = name == null ? nameI18nKey : name;
            this.value = value;
        }

        public String getName() {
            return name;
        }

        public String getNameI18nKey() {
            return nameI18nKey;
        }

        public String getValue() {
            return value;
        }
    }

    public JacksonAttributesSerializer(ObjectMapper objectMapper) {
        this.objectMapper = requireNonNull(objectMapper);
    }

    @Override
    public List<AuditAttribute> deserialize(@Nonnull String s) {
        try {
            return Stream.of(objectMapper.readValue(s, AuditAttributeData[].class))
                    .map(x -> AuditAttribute.fromI18nKeys(x.getNameI18nKey(), x.getValue()).withNameTranslation(x.getName()).build())
                    .collect(Collectors.toList());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String serialize(@Nonnull Iterable<AuditAttribute> changedValues) {
        AuditAttributeData[] values = StreamSupport.stream(changedValues.spliterator(), false)
                .map(x -> new AuditAttributeData(x.getNameI18nKey(), x.getName(), x.getValue()))
                .toArray(AuditAttributeData[]::new);
        try {
            return objectMapper.writeValueAsString(values);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
