package com.atlassian.audit.ao.dao;

import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.entity.AuditAuthor;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditResource;
import com.atlassian.audit.entity.AuditType;
import com.google.common.collect.Maps;

import java.util.List;
import java.util.Map;

import static java.util.Objects.requireNonNull;

public class AuditEntityMapper {
    private final ChangedValuesSerializer changedValueSerializer;
    private final AttributesSerializer attributesSerializer;
    private final AffectedObjectsSerializer affectedObjectsSerializer;

    public AuditEntityMapper(ChangedValuesSerializer changedValueSerializer,
                             AttributesSerializer attributesSerializer,
                             AffectedObjectsSerializer affectedObjectsSerializer) {
        this.changedValueSerializer = requireNonNull(changedValueSerializer);
        this.attributesSerializer = requireNonNull(attributesSerializer);
        this.affectedObjectsSerializer = requireNonNull(affectedObjectsSerializer);
    }

    public Map<String, Object> map(AuditEntity auditEntity) {
        Map<String, Object> values = Maps.newHashMap();
        AuditType auditType = auditEntity.getAuditType();
        values.put(AoAuditEntity.LEVEL_COLUMN, auditType.getLevel().name());
        values.put(AoAuditEntity.AREA_COLUMN, auditType.getArea().name());
        values.put(AoAuditEntity.ACTION_TKEY_COLUMN, auditType.getActionI18nKey());
        values.put(AoAuditEntity.ACTION_COLUMN, auditType.getAction());
        values.put(AoAuditEntity.CATEGORY_TKEY_COLUMN, auditType.getCategoryI18nKey());
        values.put(AoAuditEntity.CATEGORY_COLUMN, auditType.getCategory());
        values.put(AoAuditEntity.TIMESTAMP_COLUMN, auditEntity.getTimestamp().toEpochMilli());
        values.put(AoAuditEntity.METHOD_COLUMN, auditEntity.getMethod());
        values.put(AoAuditEntity.SYSTEM_COLUMN, auditEntity.getSystem());
        values.put(AoAuditEntity.NODE_COLUMN, auditEntity.getNode());
        values.put(AoAuditEntity.SOURCE_COLUMN, auditEntity.getSource());
        values.put(AoAuditEntity.CHANGE_VALUES_COLUMN, changedValueSerializer.serialize(auditEntity.getChangedValues()));
        values.put(AoAuditEntity.ATTRIBUTES_COLUMN, attributesSerializer.serialize(auditEntity.getExtraAttributes()));
        List<AuditResource> affectedObjects = auditEntity.getAffectedObjects();
        values.put(AoAuditEntity.RESOURCES_COLUMN, affectedObjectsSerializer.serialize(affectedObjects));

        int k = 0;
        for (AuditResource x : affectedObjects) {
            switch (k) {
                case 0:
                    values.put(AoAuditEntity.RESOURCE_ID_COLUMN_1, x.getId());
                    values.put(AoAuditEntity.RESOURCE_TYPE_COLUMN_1, x.getType());
                    break;
                case 1:
                    values.put(AoAuditEntity.RESOURCE_ID_COLUMN_2, x.getId());
                    values.put(AoAuditEntity.RESOURCE_TYPE_COLUMN_2, x.getType());
                    break;
                case 2:
                    values.put(AoAuditEntity.RESOURCE_ID_COLUMN_3, x.getId());
                    values.put(AoAuditEntity.RESOURCE_TYPE_COLUMN_3, x.getType());
                    break;
                case 3:
                    values.put(AoAuditEntity.RESOURCE_ID_COLUMN_4, x.getId());
                    values.put(AoAuditEntity.RESOURCE_TYPE_COLUMN_4, x.getType());
                    break;
                case 4:
                    values.put(AoAuditEntity.RESOURCE_ID_COLUMN_5, x.getId());
                    values.put(AoAuditEntity.RESOURCE_TYPE_COLUMN_5, x.getType());
                    break;
                default:
            }
            k++;
        }

        AuditAuthor author = auditEntity.getAuthor();
        values.put(AoAuditEntity.USER_ID_COLUMN, author.getId());
        values.put(AoAuditEntity.USER_NAME_COLUMN, author.getName());
        values.put(AoAuditEntity.USER_TYPE_COLUMN, author.getType());

        String searchString = createSearchString(auditType, author, affectedObjects, auditEntity.getSource());
        values.put(AoAuditEntity.SEARCH_STRING_COLUMN, searchString);

        return values;
    }

    private String createSearchString(AuditType auditType, AuditAuthor author, Iterable<AuditResource> affectedObjects, String source) {
        SearchTokenizer searchTokenizer = new SearchTokenizer();
        affectedObjects.forEach(obj -> searchTokenizer.put(obj.getName()));
        return searchTokenizer
                    .put(auditType.getActionI18nKey())
                    .put(auditType.getCategoryI18nKey())
                    .put(auditType.getAction())
                    .put(auditType.getCategory())
                    .put(author.getName())
                    .put(source)
                    .getTokenizedString();
    }
}
