package com.atlassian.audit.ao.dao;

import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditEntity;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.List;
import java.util.function.Consumer;

public interface AuditEntityDao {

    @Nonnull
    default Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                        @Nonnull PageRequest<AuditEntityCursor> pageRequest) {
        return findBy(query, pageRequest, Integer.MAX_VALUE);
    }

    /**
     * Returns a page of entities satisfying the given query and page request.
     *
     * @param query       filter the entities to be returned.
     * @param pageRequest page of entities to fetch.
     * @param scanLimit   the maximum number of entities to be scanned in the inverse insertion order,
     *                    the {@code Integer.MAX_VALUE} which means there is no limit.
     */
    @Nonnull
    Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                @Nonnull PageRequest<AuditEntityCursor> pageRequest,
                                                int scanLimit);

    /**
     * Use to search the database for AuditEntities and supply the result one by one to consumer.
     * @param auditQuery to filter the entities being returned
     * @param consumer consumes the AuditEntity
     * @param offset number of records to skip
     * @param limit maximum number of records to produce
     */
    void stream(@Nonnull AuditQuery auditQuery, @Nonnull Consumer<AuditEntity> consumer, int offset, int limit);

    void save(@Nonnull List<AuditEntity> auditEntities);

    void save(@Nonnull AuditEntity auditEntity);

    /**
     * Remove all old audit records from before a certain time
     *
     * @param before records with timestamp earlier than it will be deleted
     */
    void removeBefore(Instant before);

    int count();

    int count(@Nonnull AuditQuery auditQuery);

    /**
     *  Delete old records from DB to retain <code>limit</code> records.
     *  Remove the data in ID ASC order, because smaller IDs are inserted earlier.
     *
     * @param limit number of records to be retained.
     * @throws IllegalArgumentException if the specified limit is lower than 10,000
     */
    void retainRecent(int limit);
}
