package com.atlassian.audit.ao.dao;

import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.entity.AuditAttribute;
import com.atlassian.audit.entity.AuditAuthor;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditType;
import com.atlassian.audit.entity.ChangedValue;
import com.atlassian.audit.entity.CoverageArea;
import com.atlassian.audit.entity.CoverageLevel;

import java.time.Instant;
import java.util.Comparator;
import java.util.stream.Collectors;

import static com.atlassian.audit.entity.AuditAuthor.UNKNOWN_AUTHOR;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

public class AoAuditEntityMapper {
    private final ChangedValuesSerializer changedValueSerializer;
    private final AttributesSerializer attributesSerializer;
    private final AffectedObjectsSerializer affectedObjectsSerializer;

    public AoAuditEntityMapper(ChangedValuesSerializer changedValueSerializer,
                               AttributesSerializer attributesSerializer,
                               AffectedObjectsSerializer affectedObjectsSerializer) {
        this.changedValueSerializer = requireNonNull(changedValueSerializer);
        this.attributesSerializer = requireNonNull(attributesSerializer);
        this.affectedObjectsSerializer = requireNonNull(affectedObjectsSerializer);
    }

    public AuditEntity map(AoAuditEntity ao) {
        AuditAuthor author = AuditAuthor.builder()
                .id(ofNullable(ao.getUserId()).orElse(UNKNOWN_AUTHOR.getId()))
                .name(ao.getUsername())
                .type(ao.getUserType())
                .build();
        return AuditEntity.builder(extractType(ao))
                .id(ao.getId())
                .source(ao.getSource())
                .timestamp(Instant.ofEpochMilli(ao.getTimestamp()))
                .affectedObjects(affectedObjectsSerializer.deserialize(ao.getResources()))
                .changedValues(changedValueSerializer.deserialize(ao.getChangedValues()).stream()
                        .sorted(Comparator.comparing(ChangedValue::getKey))
                        .collect(Collectors.toList()))
                .extraAttributes(attributesSerializer.deserialize(ao.getAttributes()).stream()
                        .sorted(Comparator.comparing(AuditAttribute::getName))
                        .collect(Collectors.toList()))
                .author(author)
                .method(ao.getMethod())
                .system(ao.getSystem())
                .node(ao.getNode())
                .build();
    }

    private AuditType extractType(AoAuditEntity ao) {
        return AuditType.fromI18nKeys(CoverageArea.valueOf(ao.getArea()),
                                      CoverageLevel.valueOf(ao.getLevel()),
                                      ao.getCategoryI18nKey() == null ? ao.getCategory(): ao.getCategoryI18nKey(),
                                      ao.getActionI18nKey() == null ? ao.getAction() : ao.getActionI18nKey())
                     .withCategoryTranslation(ao.getCategory())
                     .withActionTranslation(ao.getAction())
                     .build();
    }
}
