package com.atlassian.audit.ao.dao;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import net.java.ao.Query;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;

import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.ID_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.TIMESTAMP_COLUMN;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

public class AoAuditEntityDao implements AuditEntityDao {

    // Minimal number of records to be retained
    public static final int MIN_RETAIN_LIMIT_DEFAULT = 10_000;

    private static final String DELETE_BATCH_LIMIT_KEY = "plugin.audit.db.delete.batch.limit";
    private static final int DELETE_BATCH_LIMIT_DEFAULT = 10_000;
    private final ActiveObjects ao;

    private final TransactionTemplate transactionTemplate;
    private final AuditQueryMapper auditQueryMapper;
    private final AuditEntityMapper auditEntityMapper;
    private final AoAuditEntityMapper aoAuditEntityMapper;
    private final Supplier<Integer> deleteBatchSizeSupplier;
    private final int minRetainLimit;

    public AoAuditEntityDao(ActiveObjects ao,
                            TransactionTemplate transactionTemplate,
                            AuditQueryMapper auditQueryMapper,
                            AuditEntityMapper auditEntityMapper,
                            AoAuditEntityMapper aoAuditEntityMapper,
                            PropertiesProvider propertiesProvider
    ) {
        this(ao, transactionTemplate, auditQueryMapper, auditEntityMapper, aoAuditEntityMapper,
                () -> propertiesProvider.getInteger(DELETE_BATCH_LIMIT_KEY, DELETE_BATCH_LIMIT_DEFAULT),
                MIN_RETAIN_LIMIT_DEFAULT);
    }

    @VisibleForTesting
    public AoAuditEntityDao(ActiveObjects ao,
                            TransactionTemplate transactionTemplate,
                            AuditQueryMapper auditQueryMapper,
                            AuditEntityMapper auditEntityMapper,
                            AoAuditEntityMapper aoAuditEntityMapper,
                            Supplier<Integer> deleteBatchSizeSupplier,
                            int minRetainLimit) {
        this.ao = requireNonNull(ao);
        this.transactionTemplate = transactionTemplate;

        this.auditQueryMapper = requireNonNull(auditQueryMapper);
        this.auditEntityMapper = requireNonNull(auditEntityMapper);
        this.aoAuditEntityMapper = requireNonNull(aoAuditEntityMapper);
        this.deleteBatchSizeSupplier = deleteBatchSizeSupplier;
        this.minRetainLimit = minRetainLimit;
    }

    @Nonnull
    @Override
    public Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery auditQuery, @Nonnull PageRequest<AuditEntityCursor> pageRequest, int scanLimit) {
        return transactionTemplate.execute(() -> doFindBy(auditQuery, pageRequest, scanLimit));
    }

    private Page<AuditEntity, AuditEntityCursor> doFindBy(@Nonnull AuditQuery auditQuery, @Nonnull PageRequest<AuditEntityCursor> pageRequest, int scanLimit) {
        requireNonNull(auditQuery, "auditQuery");
        requireNonNull(pageRequest, "pageRequest");

        AuditQuery scanLimitQuery;
        if (scanLimit == Integer.MAX_VALUE) {
            scanLimitQuery = AuditQuery.builder(auditQuery)
                    .build();
        } else {
            AoAuditEntity[] last = ao.find(AoAuditEntity.class, Query.select()
                    .where(format("%s >= ? AND %s <= ?", TIMESTAMP_COLUMN, TIMESTAMP_COLUMN),
                            auditQuery.getFrom().orElse(Instant.EPOCH).toEpochMilli(),
                            auditQuery.getTo().orElse(Instant.now()).toEpochMilli())
                    .limit(1)
                    .order(format("%s DESC", ID_COLUMN)));
            if (last.length == 0) {
                return Page.emptyPage();
            }
            scanLimitQuery = AuditQuery.builder(auditQuery)
                    .minId(Math.max(auditQuery.getMinId().orElse(0L), last[0].getId() - scanLimit))
                    .build();
        }

        Query query = auditQueryMapper.map(scanLimitQuery, pageRequest);
        return createPage(pageRequest, ao.find(AoAuditEntity.class, query));
    }

    private Page<AuditEntity, AuditEntityCursor> createPage(@Nonnull PageRequest<AuditEntityCursor> pageRequest, AoAuditEntity[] aoAuditEntities) {
        List<AuditEntity> entities = Arrays.stream(aoAuditEntities)
                .map(aoAuditEntityMapper::map)
                .limit(pageRequest.getLimit())
                .collect(toList());
        if (entities.isEmpty()) {
            return Page.emptyPage();
        }

        AoAuditEntity lastAuditEntity = aoAuditEntities[Math.min(pageRequest.getLimit(), aoAuditEntities.length) - 1];
        PageRequest<AuditEntityCursor> nextPageRequest = new PageRequest.Builder<AuditEntityCursor>()
                .cursor(new AuditEntityCursor(Instant.ofEpochMilli(lastAuditEntity.getTimestamp()),
                        lastAuditEntity.getId()))
                .limit(pageRequest.getLimit())
                .build();
        return new Page.Builder<AuditEntity, AuditEntityCursor>(entities,
                aoAuditEntities.length <= pageRequest.getLimit()).nextPageRequest(nextPageRequest)
                .build();
    }

    @Override
    public void stream(@Nonnull AuditQuery auditQuery, @Nonnull Consumer<AuditEntity> consumer, int offset, int limit) {
        requireNonNull(auditQuery, "auditQuery");
        requireNonNull(consumer, "consumer");
        Query query = auditQueryMapper.map(auditQuery)
                .order(format("%s DESC, %s DESC", TIMESTAMP_COLUMN, ID_COLUMN))
                .offset(offset)
                .limit(limit);
        transactionTemplate.execute(() -> {
            ao.stream(AoAuditEntity.class, query, aoAuditEntity -> consumer.accept(aoAuditEntityMapper.map(aoAuditEntity)));
            return null;
        });
    }

    @Override
    public void save(@Nonnull List<AuditEntity> auditEntities) {
        requireNonNull(auditEntities, "auditEntities");
        transactionTemplate.execute(() -> {
            ao.create(AoAuditEntity.class, auditEntities.stream()
                    .map(auditEntityMapper::map)
                    .collect(toList()));
            return null;
        });
    }

    @Override
    public void save(@Nonnull AuditEntity auditEntity) {
        requireNonNull(auditEntity, "auditEntity");
        transactionTemplate.execute(() -> ao.create(AoAuditEntity.class, auditEntityMapper.map(auditEntity)));
    }

    @Override
    public void removeBefore(Instant before) {
        int batchSize = deleteBatchSizeSupplier.get();
        int deletedCount;
        do {
            deletedCount = transactionTemplate.execute(() -> {
                AoAuditEntity[] last = ao.find(AoAuditEntity.class,
                        Query.select(format("%s, %s", TIMESTAMP_COLUMN, ID_COLUMN))
                                .where(format("%s < ?", TIMESTAMP_COLUMN), before.toEpochMilli())
                                .limit(1)
                                .offset(batchSize - 1)
                                .order(format("%s, %s", TIMESTAMP_COLUMN, ID_COLUMN)));
                if (last.length == 0) {
                    return ao.deleteWithSQL(AoAuditEntity.class,
                            format("%s < ?", TIMESTAMP_COLUMN),
                            before.toEpochMilli());
                }
                return ao.deleteWithSQL(AoAuditEntity.class,
                        format("%s <= ?", TIMESTAMP_COLUMN), last[0].getTimestamp());
            });
        } while (deletedCount > 0);
    }

    @Override
    public int count() {
        return transactionTemplate.execute(() -> ao.count(AoAuditEntity.class));
    }

    @Override
    public int count(@Nonnull AuditQuery auditQuery) {
        requireNonNull(auditQuery, "auditQuery");
        Query query = auditQueryMapper.map(auditQuery);
        return transactionTemplate.execute(() -> ao.count(AoAuditEntity.class, query));
    }

    @Override
    public void retainRecent(int limit) {
        if (limit < minRetainLimit) {
            throw new IllegalArgumentException("Invalid retain limit : " + limit);
        }
        int toBeDeleted = count() - limit;
        int batchSize = deleteBatchSizeSupplier.get();
        while (toBeDeleted > 0) {
            int deletedCount = deleteOldest(Math.min(toBeDeleted, batchSize));
            if (deletedCount == 0) {
                // To avoid infinite loop in case no records are deleted. This can happen if records are deleted by some
                // other transaction.
                break;
            }
            toBeDeleted -= deletedCount;
        }
    }

    private Integer deleteOldest(int count) {
        return transactionTemplate.execute(() -> {
            AoAuditEntity[] last = ao.find(AoAuditEntity.class,
                    Query.select(format("%s,%s", TIMESTAMP_COLUMN, ID_COLUMN))
                            .limit(1).offset(count - 1)
                            .order(format("%s, %s", TIMESTAMP_COLUMN, ID_COLUMN)));
            if (last.length == 0) {
                return 0;
            }
            return ao.deleteWithSQL(AoAuditEntity.class,
                    format("%s < ? OR (%s = ? AND %s <= ?)", TIMESTAMP_COLUMN, TIMESTAMP_COLUMN, ID_COLUMN),
                    last[0].getTimestamp(), last[0].getTimestamp(), last[0].getId());
        });
    }
}
