package com.atlassian.audit.api.util.pagination;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;

/**
 * Pagination for a request.
 */
public class PageRequest<C> {

    private final int offset;
    private final int limit;
    private final C cursor;

    private PageRequest(Builder<C> builder) {
        offset = builder.offset;
        limit = builder.limit;
        cursor = builder.cursor;
    }

    /**
     * @return the index of the element in the result set that this page will start at
     */
    public int getOffset() {
        return offset;
    }

    /**
     * Used for cursor based pagination. Cursor based pagination can be used by services to return results with better
     * performance vs offset based pagination as offset may need all results to be scanned in order to skip records.
     * When using cursor based pagination, each page of results are ordered by a unique
     * combination of field(s) which forms a cursor. Clients can request for a page that contains results starting from
     * and excluding given cursor.
     *
     * @return cursor if specified in this page request, otherwise {@link Optional#empty()}
     * @see Page#getNextPageRequest()
     */
    @Nonnull
    public Optional<C> getCursor() {
        return Optional.ofNullable(cursor);
    }

    /**
     * @return the maximum allowed size of the page
     */
    public int getLimit() {
        return limit;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PageRequest<?> that = (PageRequest<?>) o;
        return offset == that.offset &&
                limit == that.limit &&
                Objects.equals(cursor, that.cursor);
    }

    @Override
    public int hashCode() {
        return Objects.hash(offset, cursor, limit);
    }

    @Override
    public String toString() {
        return "PageRequest{" +
                "offset=" + offset +
                ", limit=" + limit +
                ", cursor=" + cursor +
                '}';
    }

    public static class Builder<C> {
        /**
         * The maximum size of any page limit.
         */
        int MAX_PAGE_LIMIT = 100000;

        private int offset;
        private int limit;
        private C cursor;

        /**
         * @param offset the number of records to skip from the actual results fetched while creating a page of results
         * @return this Builder
         */
        @Nonnull
        public Builder<C> offset(int offset) {
            this.offset = offset;
            return this;
        }

        /**
         * @param limit the maximum number of records to be fetched in resulting page.
         * @return this Builder
         */
        @Nonnull
        public Builder<C> limit(int limit) {
            this.limit = limit;
            return this;
        }

        /**
         * @param cursor cursor the cursor from where to start page of results being returned.
         * @return this Builder
         * @see PageRequest#getCursor()
         */
        @Nonnull
        public Builder<C> cursor(@Nullable C cursor) {
            this.cursor = cursor;
            return this;
        }

        /**
         * @return the {@link PageRequest} with parameters set in this Builder
         */
        @Nonnull
        public PageRequest<C> build() {
            if (limit > MAX_PAGE_LIMIT) {
                throw new IllegalArgumentException("Limit must be less than " + MAX_PAGE_LIMIT);
            }
            return new PageRequest<>(this);
        }
    }
}
