package com.atlassian.audit.api.util.pagination;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;

/**
 * Provides support for pagination.
 *
 * @param <T> the type of the entity
 * @param <C> represents the type of cursor location to be used for pagination. See {@link PageRequest#getCursor()}
 */
public class Page<T, C> {

    private final List<T> values;
    private final PageRequest<C> nextPageRequest;

    private Page(Builder<T, C> builder) {
        this.values = builder.values;
        this.nextPageRequest = builder.nextPageRequest;
    }

    /**
     * @return {@code true} if there are no more results; otherwise, {@code false} if at least one more page,
     * perhaps partially filled but not empty, of results is available
     */
    public boolean getIsLastPage() {
        return nextPageRequest == null;
    }

    /**
     * @return a request which can be used to retrieve the next page, which will be {@link Optional#empty() empty} if
     * this was the {@link #getIsLastPage() last page}
     */
    @Nonnull
    public Optional<PageRequest<C>> getNextPageRequest() {
        return Optional.ofNullable(nextPageRequest);
    }

    /**
     * @return the number of results in the page
     */
    public int getSize() {
        return values.size();
    }

    /**
     * @return the page's results
     */
    @Nonnull
    public List<T> getValues() {
        return values;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Page<?, ?> page = (Page<?, ?>) o;
        return values.equals(page.values) &&
                Objects.equals(nextPageRequest, page.nextPageRequest);
    }

    @Override
    public int hashCode() {
        return Objects.hash(values, nextPageRequest);
    }

    public static class Builder<T, C> {
        private PageRequest<C> nextPageRequest;
        private boolean lastPage;
        private final List<T> values;

        /**
         * Create builder for a page corresponding to given result
         *
         * @param values      values fetched for this page.
         * @param lastPage    true, if this is the last page of results, otherwise false
         */
        public Builder(@Nonnull List<T> values, boolean lastPage) {
            this.values = unmodifiableList(requireNonNull(values));
            this.lastPage = lastPage;
        }

        /**
         * @param nextPageRequest the {@link PageRequest page request} that can be used to fetch next page in sequence.
         *                        This field is mandatory if this is not a {@link Page#getIsLastPage() last page}
         * @return this Builder
         */
        @Nonnull
        public Builder<T, C> nextPageRequest(PageRequest<C> nextPageRequest) {
            this.nextPageRequest = nextPageRequest;
            return this;
        }

        @Nonnull
        public Page<T, C> build() {
            if (!lastPage) {
                requireNonNull(nextPageRequest, "nextPageRequest should be non-null except for last page");
            } else {
                nextPageRequest = null;
            }
            return new Page<>(this);
        }
    }

    public static <T, C> Page<T, C> emptyPage() {
        return new Page.Builder<T, C>(Collections.emptyList(), true).build();
    }
}
