package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Represents the Audit Type information of the Audit event including
 * Audit Area
 * Audit Level
 * Audit category
 * Audit action
 * <p>
 * Audit event types are unique across applications.
 */
public class AuditType {

    private final CoverageArea area;

    private final String category;

    private final String categoryI18nKey;

    private final String action;

    private final String actionI18nKey;

    private final CoverageLevel level;

    private AuditType(Builder builder) {
        this.area = requireNonNull(builder.area);
        this.categoryI18nKey = requireNonNull(builder.categoryI18nKey);
        this.actionI18nKey = requireNonNull(builder.actionI18nKey);
        this.category = builder.category;
        this.action = builder.action;
        this.level = requireNonNull(builder.level);
    }

    /**
     * Audit Type
     * @deprecated since 1.7.0 use {@link #AuditType(Builder)} variant which expects i18n keys
     * for category and action.
     */
    @Deprecated
    public AuditType(@Nonnull CoverageArea area,
                     @Nonnull String category,
                     @Nonnull String action,
                     @Nonnull CoverageLevel level) {
        this.area = requireNonNull(area);
        this.categoryI18nKey = requireNonNull(category);
        this.actionI18nKey = requireNonNull(action);
        this.category = requireNonNull(category);
        this.action = requireNonNull(action);
        this.level = level;
    }

    @Nonnull
    public CoverageArea getArea() {
        return area;
    }

    /**
     * @return the i18n key of the category of the event, e.g. audit.product.category.projects
     */
    @Nonnull
    public String getCategoryI18nKey() {
        return categoryI18nKey;
    }

    /**
     * @return the category of the event translated, e.g. "Projects"
     */
    @Nullable
    public String getCategory() {
        return category;
    }

    /**
     * @return the i18n key of the scenario of the event, e.g. audit.product.action.projects.create
     */
    @Nonnull
    public String getActionI18nKey() {
        return actionI18nKey;
    }

    /**
     * @return the scenario of the event, e.g. "Create Project"
     */
    @Nullable
    public String getAction() {
        return action;
    }

    @Nonnull
    public CoverageLevel getLevel() {
        return level;
    }

    @Override
    public String toString() {
        return "AuditType{" +
                "area=" + area +
                ", categoryI18nKey='" + categoryI18nKey + '\'' +
                ", category='" + category + '\'' +
                ", actionI18nKey='" + actionI18nKey + '\'' +
                ", action='" + action + '\'' +
                ", level=" + level +
                '}';
    }

    public static Builder fromI18nKeys(@Nonnull CoverageArea area,
                                       @Nonnull CoverageLevel level,
                                       @Nonnull String categoryI18nKey,
                                       @Nonnull String actionI18nKey) {
        return new Builder(area, level, categoryI18nKey, actionI18nKey);
    }

    public static Builder builder(@Nonnull AuditType auditType) {
        return new Builder(auditType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditType auditType = (AuditType) o;
        return area == auditType.area &&
                categoryI18nKey.equals(auditType.categoryI18nKey) &&
                Objects.equals(category, auditType.category) &&
                actionI18nKey.equals(auditType.actionI18nKey) &&
                Objects.equals(action, auditType.action) &&
                level == auditType.level;
    }

    @Override
    public int hashCode() {
        return Objects.hash(area, category, categoryI18nKey, action, actionI18nKey, level);
    }

    public static class Builder {
        private CoverageArea area;
        private String category;
        private String categoryI18nKey;
        private String action;
        private String actionI18nKey;
        private CoverageLevel level;

        private Builder(@Nonnull CoverageArea area,
                        @Nonnull CoverageLevel level,
                        @Nonnull String categoryI18nKey,
                        @Nonnull String actionI18nKey) {
            this.area = requireNonNull(area);
            this.categoryI18nKey = requireNonNull(categoryI18nKey);
            this.actionI18nKey = requireNonNull(actionI18nKey);
            this.level = requireNonNull(level);
        }

        private Builder(AuditType auditType) {
            this.area = requireNonNull(auditType.getArea());
            this.categoryI18nKey = requireNonNull(auditType.getCategoryI18nKey());
            this.actionI18nKey = requireNonNull(auditType.getActionI18nKey());
            this.category = auditType.getCategory();
            this.action = auditType.getAction();
            this.level = requireNonNull(auditType.getLevel());
        }

        public Builder withCategoryTranslation(String category) {
            this.category = category;
            return this;
        }

        public Builder withActionTranslation(String action) {
            this.action = action;
            return this;
        }

        public AuditType build() {
            return new AuditType(this);
        }
    }
}
