package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

/**
 * Represents a resource affected by an auditing event, used in {@link AuditEntity#getAffectedObjects()}
 * <p>
 * E.g.
 * a Confluence page, a Jira issue, a Bitbucket repo
 */
public class AuditResource {

    private final String name;

    private final String type;

    private String uri;

    private String id;

    private AuditResource(Builder builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.uri = builder.uri;
        this.type = builder.type;
    }

    /**
     * @return the unique name of the resource.
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return the type of the resource, e.g. Issue.
     */
    @Nonnull
    public String getType() {
        return type;
    }

    /**
     * @return the uri reference of the resource.
     */
    @Nullable
    public String getUri() {
        return uri;
    }

    /**
     * @return id of the resource
     */
    @Nullable
    public String getId() {
        return id;
    }

    @Override
    public String toString() {
        return "AuditResource{" +
                "name='" + name + '\'' +
                ", type='" + type + '\'' +
                ", uri='" + uri + '\'' +
                ", id='" + id + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditResource that = (AuditResource) o;
        return name.equals(that.name) &&
                type.equals(that.type) &&
                Objects.equals(uri, that.uri) &&
                Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, type, uri, id);
    }

    public static Builder builder(@Nonnull final String name,
                                  @Nonnull final String type) {
        return new Builder(name, type);
    }

    public static class Builder {

        private String name;

        private String type;

        private String uri;

        private String id;

        private Builder(final String name,
                        final String type) {
            this.name = name;
            this.type = type;
        }

        public Builder(AuditResource resource) {
            this.id = resource.id;
            this.name = resource.name;
            this.type = resource.type;
            this.uri = resource.uri;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder type(String type) {
            this.type = type;
            return this;
        }

        public Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public AuditResource build() {
            Objects.requireNonNull(name);
            Objects.requireNonNull(type);
            return new AuditResource(this);
        }
    }
}
