package com.atlassian.audit.api;

import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditResource;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;

/**
 * Represents fields to be used for filtering {@link AuditEntity}. Following fields can be used in query -
 * <ul>
 * <li>actions - Matches all the entities that have one of these actions. If not specified this filter will
 * not be used</li>
 * <li>categories - Matches all the entities that have one of these categories. If not specified this filter will
 * not be used</li>
 * <li>from - Matches all the entities logged starting from this time. Default value is 7 days ago</li>
 * <li>to - Matches all the entities logged until this time. Default value is current time</li>
 * <li>limit - Maximum number of records to return. Default value is 100</li>
 * <li>offset - Number of matched records to skip. Default value is 0</li>
 * <li>resourceId and resourceType - Matches all the entities that have affectedObject of given ID and type. If not
 * specified this filter will not be used</li>
 * <li>userIds - Matches all the entities that have user ID belonging to one of these IDs. If not specified this
 * filter will not be used </li>
 * </ul>
 */
public class AuditQuery {

    private final Set<String> actions;
    private final Set<String> categories;
    private final Instant from;
    private final Instant to;
    private final List<AuditResourceIdentifier> resources;
    private final Set<String> userIds;
    private final String searchText;
    private final Long minId;
    private final Long maxId;

    private AuditQuery(Builder builder) {
        this.actions = builder.actions;
        this.categories = builder.categories;
        this.from = builder.from;
        this.to = builder.to;
        this.resources = builder.resources;
        this.userIds = builder.userIds;
        this.searchText = builder.searchText;
        this.minId = builder.minId;
        this.maxId = builder.maxId;
    }

    /**
     * @return the set of actions that should match {@link AuditEntity} to be queried. Empty list means this filter is
     * not applied.
     */
    @Nonnull
    public Set<String> getActions() {
        return actions;
    }

    /**
     * @return the set of categories that should match {@link AuditEntity} to be queried. Empty list means this filter is
     * not applied.
     */
    @Nonnull
    public Set<String> getCategories() {
        return categories;
    }

    /**
     * @return the start time from which {@link AuditEntity} needs to be queried.
     */
    @Nonnull
    public Optional<Instant> getFrom() {
        return Optional.ofNullable(from);
    }

    /**
     * @return the end time till which {@link AuditEntity} needs to be queried.
     */
    @Nonnull
    public Optional<Instant> getTo() {
        return Optional.ofNullable(to);
    }

    /**
     * @return the ID of the first {@link AuditResource} returned by {@link #getResources()}
     * @deprecated since release 1.5.0, replaced by {@link #getResources}
     */
    @Deprecated
    @Nonnull
    public Optional<String> getResourceId() {
        return resources.stream().findFirst().map(AuditResourceIdentifier::getId);
    }

    /**
     * @return the type of the first {@link AuditResource} returned by {@link #getResources()}
     * @deprecated since release 1.5.0, replaced by {@link #getResources}
     */
    @Deprecated
    @Nonnull
    public Optional<String> getResourceType() {
        return resources.stream().findFirst().map(AuditResourceIdentifier::getType);
    }

    /**
     * The DB query will be conducted as:
     * <ul>
     *   <li>AND between different resource types</li>
     *   <li>OR with the same resource type</li>
     * </ul>
     * For example, given
     * <ul>
     *   <li>Project P1
     *     <ul>
     *      <li>Repository R11</li>
     *      <li>Repository R12</li>
     *     </ul>
     *   </li>
     *   <li>Project P2
     *     <ul>
     *      <li>Repository R21</li>
     *      <li>Repository R22</li>
     *     </ul>
     *   </li>
     * </ul>
     * then
     * <ul>
     *   <li>the resource filter [Project:P1] returns events belong to [R11, R12]</li>
     *   <li>the resource filter [Project:P1,Repository:R11] returns events belong to [R11]</li>
     *   <li>the resource filter [Project:P1,Repository:R11,Repository:R12] returns events belong to [R11,R12]</li>
     *   <li>the resource filter [Project:P1,Project:P2,Repository:R11,Repository:R12] returns events belong to [R11,R12]</li>
     *   <li>the resource filter [Project:P1,Project:P2,Repository:R11,Repository:R12, Repository:R21] returns events belong to [R11,R12,R21]</li>
     *   <li>the resource filter [Project:P1,Repository:R21,Repository:R22] returns no result</li>
     * </ul>
     *
     * @return a set of {@link AuditResource}'s. At least one of them should be present in
     * {@link AuditEntity#getAffectedObjects()} for that entity to be found.
     */
    @Nonnull
    public Set<AuditResourceIdentifier> getResources() {
        return new HashSet<>(resources);
    }

    /**
     * @return the set of user IDs that should match {@link AuditEntity#getAuthor()} to be queried. Empty list means this
     * filter is not applied.
     */
    @Nonnull
    public Set<String> getUserIds() {
        return userIds;
    }

    /**
     * @return the text that need to be matched against action, category, resource name, user name or IP address in
     * {@link AuditEntity}
     */
    @Nonnull
    public Optional<String> getSearchText() {
        return Optional.ofNullable(searchText);
    }

    /**
     * @return the inclusive lower limit of Id, only applicable for DB storage
     */
    @Nonnull
    public Optional<Long> getMinId() {
        return Optional.ofNullable(minId);
    }

    /**
     * @return the inclusive upper limit of Id, only applicable for DB storage
     */
    @Nonnull
    public Optional<Long> getMaxId() {
        return Optional.ofNullable(maxId);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(AuditQuery query) {
        return new Builder(query);
    }

    /**
     * @return whether any filters have been applied
     */
    public boolean hasFilter() {
        return !this.getActions().isEmpty() ||
                !this.getCategories().isEmpty() ||
                this.getFrom().isPresent() ||
                this.getTo().isPresent() ||
                !this.getResources().isEmpty() ||
                !this.getUserIds().isEmpty() ||
                this.getSearchText().isPresent();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditQuery query = (AuditQuery) o;
        return actions.equals(query.actions) &&
                categories.equals(query.categories) &&
                Objects.equals(from, query.from) &&
                Objects.equals(to, query.to) &&
                resources.equals(query.resources) &&
                userIds.equals(query.userIds) &&
                Objects.equals(searchText, query.searchText) &&
                Objects.equals(minId, query.minId) &&
                Objects.equals(maxId, query.maxId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(actions, categories, from, to, resources, userIds, searchText, minId, maxId);
    }

    public static class Builder {
        private Set<String> actions = new HashSet<>();
        private Set<String> categories = new HashSet<>();
        private List<AuditResourceIdentifier> resources = new ArrayList<>();
        private Set<String> userIds = new HashSet<>();
        private Instant from;
        private Instant to;
        private String searchText;
        private Long minId;
        private Long maxId;

        private Builder() {
        }

        private Builder(AuditQuery query) {
            actions = query.getActions();
            categories = query.getCategories();
            resources = query.resources;
            userIds = query.getUserIds();
            from = query.getFrom().orElse(null);
            to = query.getTo().orElse(null);
            searchText = query.getSearchText().orElse(null);
            minId = query.getMinId().orElse(null);
            maxId = query.getMaxId().orElse(null);
        }

        public Builder actions(String... actions) {
            if (actions != null && actions.length > 0) {
                this.actions = new HashSet<>(asList(actions));
            }
            return this;
        }

        public Builder from(@Nullable Instant from) {
            this.from = from;
            return this;
        }

        public Builder to(@Nullable Instant to) {
            this.to = to;
            return this;
        }

        public Builder categories(String... categories) {
            if (categories != null && categories.length > 0) {
                this.categories = new HashSet<>(asList(categories));
            }
            return this;
        }

        /**
         * Replaces the resource filter to single resource with given {@code type} and {@code id}
         */
        public Builder resource(@Nonnull String type, @Nonnull String id) {
            requireNonNull(type, "type");
            requireNonNull(id, "id");
            this.resources.add(new AuditResourceIdentifier(type, id));
            return this;
        }

        public Builder resources(@Nonnull List<AuditResourceIdentifier> resources) {
            requireNonNull(resources, "resources");
            this.resources = new ArrayList<>(resources);
            return this;
        }

        public Builder userIds(String... userIds) {
            if (userIds != null && userIds.length > 0) {
                this.userIds = new HashSet<>(asList(userIds));
            }
            return this;
        }

        public Builder searchText(@Nullable String searchText) {
            this.searchText = searchText;
            return this;
        }

        public Builder minId(@Nonnull Long minId) {
            this.minId = requireNonNull(minId, "minId");
            return this;
        }

        public Builder maxId(@Nonnull Long maxId) {
            this.maxId = requireNonNull(maxId, "maxId");
            return this;
        }

        public AuditQuery build() {
            return new AuditQuery(this);
        }
    }

    public static class AuditResourceIdentifier {

        @Nonnull
        private final String type;

        @Nonnull
        private final String id;

        public AuditResourceIdentifier(@Nonnull String type, @Nonnull String id) {
            this.type = requireNonNull(type, "type");
            this.id = requireNonNull(id, "id");
        }

        @Nonnull
        public String getType() {
            return type;
        }

        @Nonnull
        public String getId() {
            return id;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AuditResourceIdentifier that = (AuditResourceIdentifier) o;
            return type.equals(that.type) &&
                    id.equals(that.id);
        }

        @Override
        public int hashCode() {
            return Objects.hash(type, id);
        }

        @Override
        public String toString() {
            return "{" +
                    "type='" + type + '\'' +
                    ", id='" + id + '\'' +
                    '}';
        }
    }
}
