package com.atlassian.audit.entity;

import java.util.AbstractMap;
import java.util.Map;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toMap;

public enum EffectiveCoverageLevel {

    OFF("off", 0),

    BASE("base", 1),

    ADVANCED("advanced", 2),

    FULL("full", 3);

    private static final Map<CoverageLevel, Integer> eventLevel =
            Stream.of(new AbstractMap.SimpleEntry<>(CoverageLevel.BASE, 1),
                    new AbstractMap.SimpleEntry<>(CoverageLevel.ADVANCED, 2),
                    new AbstractMap.SimpleEntry<>(CoverageLevel.FULL, 3))
                    .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));

    private final String key;

    private final int effectiveLevel;

    EffectiveCoverageLevel(String key, int effectiveLevel) {
        this.key = key;
        this.effectiveLevel = effectiveLevel;
    }

    /**
     * Returns the most restrictive level between {@code this} and {@code other}.
     * For example {@code BASE.min(ADVANCED) = BASE}.
     */
    public EffectiveCoverageLevel mostRestrictive(EffectiveCoverageLevel other) {
        if (this.effectiveLevel < other.effectiveLevel) {
            return this;
        }
        return other;
    }

    public boolean shouldAllow(CoverageLevel entityLevel) {
        return effectiveLevel >= eventLevel.get(entityLevel);
    }

    public String getKey() {
        return key;
    }

    @Override
    public String toString() {
        return key;
    }

    public static EffectiveCoverageLevel fromKey(String key) {
        return Stream.of(EffectiveCoverageLevel.values())
                .filter(c -> c.key.equals(key))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("No such value found: " + key));
    }

}
