package com.atlassian.audit.api;

import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditEntity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

/**
 * Potential API
 * <p>
 * Implement to allow inspection of the {@link AuditEntity}s that have been consumed.
 * <p>
 * As per {@link AuditConsumer} it is expected that we will have multiple implementations eg:
 * inspect entities recorded in a relational database
 * inspect entities recorded in a file
 */
public interface AuditSearchService {

    /**
     * Use to find {@link AuditEntity}s that have been consumed
     *
     * @param query       filter the entities to be returned.
     * @param pageRequest specifies which page of entities to be returned.
     */
    @Nonnull
    default Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                        @Nonnull PageRequest<AuditEntityCursor> pageRequest) throws TimeoutException {
        return findBy(query, pageRequest, Integer.MAX_VALUE);
    }

    /**
     * Use to find {@link AuditEntity}s that have been consumed while protecting data store from being overloaded by
     * expensive queries.
     *
     * @param query       filter the entities to be returned.
     * @param pageRequest specifies which page of entities to be returned.
     * @param scanLimit   the maximum number of entities to be scanned in the inverse insertion order,
     *                    the Integer.MAX_VALUE which means there is no limit.
     */
    @Nonnull
    Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                @Nonnull PageRequest<AuditEntityCursor> pageRequest,
                                                int scanLimit) throws TimeoutException;

    /**
     * Searches database for AuditEntities and feeds the results one by one to the consumer
     * @param query to filter the entities being returned
     * @param offset number of records to skip
     * @param limit maximum number of records to produce
     * @param consumer consumes the AuditEntity
     */
    void stream(@Nonnull AuditQuery query, int offset, int limit, @Nonnull Consumer<AuditEntity> consumer) throws TimeoutException;

    /**
     * @param query filter used to search entities, null filter will count all entities
     * @return number of entities match the filter
     */
    long count(@Nullable AuditQuery query) throws TimeoutException;

}
