package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

/**
 * Represents author of an auditing event, used in {@link AuditEntity#getAuthor()}
 */
public class AuditAuthor {

    public static final AuditAuthor SYSTEM_AUTHOR = builder().id("-1").type("system").name("System").build();

    public static final AuditAuthor ANONYMOUS_AUTHOR = builder().id("-2").type("user").name("Anonymous").build();

    public static final AuditAuthor UNKNOWN_AUTHOR = builder().id("-3").type("user").name("Unknown").build();

    private final String id;

    private final String type;

    private String name;

    private String uri;

    private AuditAuthor(Builder builder) {
        this.id = builder.id;
        this.type = builder.type;
        this.name = builder.name;
        this.uri = builder.uri;
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * @return Unique key identify of the author, usually the user acoount.
     */
    @Nonnull
    public String getId() {
        return id;
    }

    /**
     * @return the type of the author, e.g. Admin.
     */
    @Nonnull
    public String getType() {
        return type;
    }

    /**
     * @return the readable name of the author.
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * @return a uri reference of the author.
     */
    @Nullable
    public String getUri() {
        return uri;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditAuthor that = (AuditAuthor) o;
        return id.equals(that.id) &&
                Objects.equals(type, that.type) &&
                Objects.equals(uri, that.uri);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, type);
    }

    @Override
    public String toString() {
        return "AuditAuthor{" +
                "key='" + id + '\'' +
                ", type='" + type + '\'' +
                ", name='" + name + '\'' +
                ", uri='" + uri + '\'' +
                '}';
    }

    public static class Builder {

        private String id;

        private String type;

        private String name;

        private String uri;

        public Builder() {
        }

        public Builder(AuditAuthor author) {
            this.id = author.id;
            this.type = author.type;
            this.name = author.name;
            this.uri = author.uri;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder type(String type) {
            this.type = type;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public AuditAuthor build() {
            Objects.requireNonNull(id);
            Objects.requireNonNull(type);
            return new AuditAuthor(this);
        }
    }
}
