package com.atlassian.applinks.internal.rest.model.status;

import com.atlassian.applinks.internal.common.rest.model.applink.RestMinimalApplicationLink;
import com.atlassian.applinks.internal.rest.model.BaseRestEntity;
import com.atlassian.applinks.internal.rest.model.ReadOnlyRestRepresentation;
import com.atlassian.applinks.internal.status.ApplinkStatus;
import org.codehaus.jackson.map.annotate.JsonSerialize;

import java.net.URI;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * JSON entity representing {@code ApplinkStatus}. This is a read-only entity (not used for updates).
 *
 * @since 4.3
 */
@JsonSerialize
public class RestApplinkStatus extends BaseRestEntity implements ReadOnlyRestRepresentation<ApplinkStatus> {
    public static final String LINK = "link";
    public static final String WORKING = "working";
    public static final String ERROR = "error";

    public static final String LOCAL_AUTHENTICATION = "localAuthentication";
    public static final String REMOTE_AUTHENTICATION = "remoteAuthentication";

    @SuppressWarnings("ConstantConditions")
    public RestApplinkStatus(@Nonnull ApplinkStatus status) {
        this(status, null);
    }

    @SuppressWarnings("ConstantConditions")
    public RestApplinkStatus(@Nonnull ApplinkStatus status, @Nullable URI authorisationCallback) {
        checkNotNull(status, "status");
        put(LINK, new RestMinimalApplicationLink(status.getLink()));
        put(WORKING, status.isWorking());
        put(LOCAL_AUTHENTICATION, new RestApplinkOAuthStatus(status.getLocalAuthentication()));
        putAs(REMOTE_AUTHENTICATION, status.getRemoteAuthentication(), RestApplinkOAuthStatus.class);
        if (!status.isWorking()) {
            put(ERROR, status.getError().accept(new RestApplinkError.Visitor(authorisationCallback)));
        }
    }
}
