package com.atlassian.applinks.analytics;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.applinks.api.ApplicationType;
import com.atlassian.applinks.internal.status.ApplinkStatus;
import com.atlassian.applinks.internal.status.error.ApplinkError;
import com.atlassian.applinks.spi.application.IdentifiableType;

import javax.annotation.Nonnull;

@EventName("applinks.detailed.status")
public class ApplinksStatusEvent {
    protected static final String WORKING_STATUS = "working";

    private final String type;
    private final String applicationId;
    private final String status;
    private final String category;

    /* Authentication Types */
    // the reason why we are making them as a field each instead of a single field with their permutations is
    // because whitelist needs exact string match for the field value. That is, we would need to declare in whitelist
    // "basic trusted", "other basic oauthTwoLO", "basic trusted cors other", etc. We would need 127 exact strings.
    private final boolean basic;
    private final boolean trusted;
    private final boolean other;
    private final boolean cors;
    private final boolean oauthTwoLO;
    private final boolean oauthTwoLOi;
    private final boolean oauthThreeLO;
    // add more types here if required.

    private ApplinksStatusEvent(final Builder builder) {
        this.applicationId = builder.applicationId;
        this.status = builder.status;
        this.category = builder.category;
        this.type = builder.type;

        this.basic = builder.basic;
        this.trusted = builder.trusted;
        this.cors = builder.cors;
        this.oauthTwoLO = builder.twoLo;
        this.oauthTwoLOi = builder.twoLoi;
        this.oauthThreeLO = builder.threeLo;
        this.other = builder.other;
    }

    public String getApplicationId() {
        return applicationId;
    }

    public String getType() {
        return type;
    }

    public String getStatus() {
        return status;
    }

    public String getCategory() {
        return category;
    }

    public boolean isBasic() {
        return basic;
    }

    public boolean isTrusted() {
        return trusted;
    }

    public boolean isOther() {
        return other;
    }

    public boolean isCors() {
        return cors;
    }

    public boolean isOauthTwoLO() {
        return oauthTwoLO;
    }

    public boolean isOauthTwoLOi() {
        return oauthTwoLOi;
    }

    public boolean isOauthThreeLO() {
        return oauthThreeLO;
    }

    public static class Builder {
        private final String applicationId;
        private final String type;
        private final String status;
        private final String category;

        public boolean basic;
        public boolean trusted;
        public boolean cors;
        public boolean twoLo;
        public boolean twoLoi;
        public boolean threeLo;
        public boolean other;

        public Builder(@Nonnull final ApplinkStatus applinkStatus) {
            applicationId = applinkStatus.getLink().getId().get();
            ApplicationType appType = applinkStatus.getLink().getType();
            type = appType instanceof IdentifiableType ? ((IdentifiableType) appType).getId().get() : appType.getI18nKey();
            ApplinkError error = applinkStatus.getError();
            if (error == null) {
                status = WORKING_STATUS;
                category = WORKING_STATUS;
            } else {
                status = error.getType().name().toLowerCase();
                category = error.getType().getCategory().name().toLowerCase();
            }
        }

        public ApplinksStatusEvent build() {
            return new ApplinksStatusEvent(this);
        }
    }
}
