(function (_, Backbone, templateNamespace, $) {

    AJS.$.namespace('Agent.Applinks.Table');
    var baseUrl = location.protocol + '//' + location.host + AJS.contextPath();

    /**
     * Main table backbone view, handles data
     * loading, remove and updating.
     */
    Agent.Applinks.Table = Backbone.View.extend({

        template: templateNamespace,

        initialize: function (options) {
            // extend with options
            _.extend(this, options);

            this.collection = new Agent.Applinks.TableCollection();
            this.$el.html(this.template.table({
                baseUrl: baseUrl
            }));
            this.registerEvents();

            // initial data load
            this.collection.fetch().complete(
                _.bind(function() {
                    this.onCollectionLoad();
                    this.onCollectionUpdate();
                }, this)
            );

            // fired when page is refreshed by the user
            AJS.$(window).on('beforeunload', function () {
                window.isPageBeingRefreshed = true;
            });
        },

        /**
         * Placeholder for registered events.
         */
        registerEvents: function () {
            var t = this;
            // collection events
            this.listenTo(this.collection, 'change remove',
                _.bind(_.throttle(this.onCollectionUpdate, 200, {
                    trailing: true
                }), this)
            );

            // edit form dialog
            this.$el.on('click', 'a.edit',
                _.bind(this.onFormEdit, this)
            );

            this.$el.on('mouseenter', 'tr', function() {
                t.promotePendingId($(this).attr('id'));
            })
                .on('click', '.block-icon', function() {
                    var $t = $(this).closest('tr');
                    var id = $t.attr('id');
                    t.onFetchStatus([id], true);
                });

            // legacy/advance call
            this.$el.on('click', 'a.advance',
                _.bind(function() {
                    var $row = AJS.$(event.target).parents('tr:first');
                    this.onAdvanceDialog($row.attr('id'));
                }, this)
            );

            // dropdown events
            AJS.$(document).on('click', '.aui-dropdown2 li',
                _.bind(this.onDropdownClick, this)
            );


            // escape pressed handler
            AJS.$(document).on('keydown', _.bind(function(event) {
                if (event.keyCode === 27) {
                    this.onFormCancel(event);
                }
            }, this));
        },

        /**
         * Executed when collection is fetched.
         */
        onCollectionLoad: function () {
            var hasData = this.collection.length;
            var table = this.$el.find("#agent-table");

            this.untooltips();

            // final checks
            table.find('thead').toggleClass('hidden', !hasData);

            if (!hasData) {
                // add empty placeholder
                table.find('tbody').empty().append(this.template.rowStatic({
                    content: AJS.I18n.getText('applinks.agent.empty'),
                    showSpinner: false
                }));
            }
            else {
                // collect all ids and fetch status
                this.onFetchStatus(this.collection.map(_.bind(function(model) {
                    return model.get('id');
                }, this)));
            }

            this.renderTooltips();
            table.addClass("applinks-loaded");
        },

        /**
         * Executed when collection is updated.
         */
        onCollectionUpdate: function () {
            var rows = document.createDocumentFragment();
            var $container = this.$el.find('tbody');

            this.untooltips($container);

            // generate rows items
            this.collection.each(_.bind(function (model) {
                rows.appendChild(new Agent.Applinks.TableRow({
                    levels: this.levels,
                    model: model
                }).render().get(0))
            }, this));

            // clear container
            $container.empty();

            // update table content (in one go)
            rows.childNodes.length ?
                $container.append(rows) :
                $container.append(this.template.rowStatic({
                    content: AJS.I18n.getText('applinks.agent.empty'),
                    showSpinner: false
                }));

            $container
                .find('.in-progress .progress-spinner')
                .spin();

            this.renderTooltips($container);
        },

        /**
         * Show inline form dialog.
         */
        onFormEdit: function (event) {
            var $row = AJS.$(event.target).parents('tr:first');
            var id = $row.attr('id');

            // stop current event
            event.preventDefault();
            event.stopPropagation();

            // requested dialog is already opened
            if (AJS.InlineDialog.current && AJS.InlineDialog.current.id === id) {
                return;
            }

            AJS.InlineDialog.current && AJS.InlineDialog.current.hide();

            // reset active row
            $row.parent().find('tr').removeClass('active');
            $row.addClass('active');
            var dialog = $row.data('dialog');

            if (!dialog) {

                var dialogId = 'agent-table-dialog-' + id;
                $('#inline-dialog-' + dialogId).remove();


                // configuration
                var dialogConfig = {
                    container: 'body',
                    hideDelay: null,
                    width: 480
                };

                var dialog = AJS.InlineDialog($row.find('td.actions .edit'), dialogId,
                    _.bind(function ($content, trigger, showPopup) {
                        var form = new Agent.Applinks.TableForm({
                            model: this.collection.get(id),
                            levels: this.levels,
                            baseUrl: baseUrl
                        });

                        // listen for save and cancel event
                        this.listenTo(form, 'save', function (event, id) {
                            this.onFormSave(event, id);
                        })
                            .listenTo(form, 'cancel', function (event) {
                                this.onFormCancel(event);
                            });

                        // render form & reapply tooltips
                        $content.html(form.render());
                        this.renderTooltips($content);

                        showPopup();
                    }, this), dialogConfig
                );

                // needed for JIRA - remove any custom layer handling
                dialog.on('click', function (event) {
                    event.preventDefault();
                    event.stopPropagation();
                });

                $row.data('dialog', dialog);
            }

            dialog.show();
        },

        /**
         * Form edit was canceled.
         */
        onFormCancel: function (event) {
            // remove any existing tipsy element
            AJS.$('body > .tipsy').remove();

            if (AJS.InlineDialog.current) {
                AJS.InlineDialog.current.hide();
            }

            this.$el.find('tr.active').removeClass('active');

            if (event) {
                event.preventDefault();
            }

            return this;
        },

        /**
         * Form was submitted. We pass in error callback,
         * which injects form & field errors.
         */
        onFormSave: function (event, id) {
            var $dialog = AJS.InlineDialog.current.popup;

            // disable all field-groups & show loading icon
            $dialog
                .find('fieldset:first').addClass('disabled').end()
                .find('.aui-button.save').attr('disabled', 'disabled').end()
                .find('.aui-icon-wait').removeClass('hidden');

            // success and error callback definition
            var callbacks = {
                success: _.bind(function(model) {
                    $dialog.hide();
                    this.onFetchStatus([model.get('id')], true);
                }, this),
                error: _.bind(function(model, response) {

                    var $form = $dialog.find('form:visible');
                    try {
                        var data = AJS.$.parseJSON(response.responseText);
                    } catch (ex) { }

                    // remove any errors
                    $dialog.find('div.error').remove();

                    // any form errors?
                    if (response.status == 0) {
                        $form.find('.form-errors').append(this.template.formError({
                            message: AJS.I18n.getText("applinks.agent.form.update.timeout")
                        }));
                    } else if (data) {
                        if (data.errors && data.errors.length) {
                            $form.prepend(this.template.formError({
                                message: AJS.$.trim(data.errors.join(' '))
                            }));
                        }

                        // inject any field errors
                        _.each(data.fieldErrors, _.bind(function (errors, key) {
                            var $field = $form.find('input[name="' + key + '"]');

                            $field.parent().append(this.template.formFieldError({
                                message: AJS.$.trim(errors.join(' '))
                            }));
                        }, this));

                    } else {
                        if (typeof response.responseText == 'string') {
                            $form.find('.form-errors').append(this.template.formError({
                                message: response.responseText
                            }));
                        }
                    }

                    // re-enable all field-groups & hide loading icon
                    $dialog.find('.aui-icon-wait').addClass('hidden');

                    // reset current popup position (because errors were added)
                    AJS.InlineDialog.current.reset();
                }, this)
            };

            // stop current event
            event.preventDefault();
            event.stopPropagation();

            // update applinks information
            this.collection.get(id).saveFormValues(
                AJS.$(event.target).parents('form:first'), callbacks
            );

            return this;
        },

        /**
         * Drop-down menu was clicked.
         */
        onDropdownClick: function(event) {
            var $element = AJS.$(event.target);
            var id = $element.parents('.aui-dropdown2:first').data('id');

            // trigger selected action
            switch ($element.data('action')) {
                case 'delete'  : this.onDeleteConfirmation(id); break;
                case 'primary' : this.onMakePrimary(id);        break;
                case 'advance' : this.onAdvanceDialog(id);      break;
                case 'remote'  : this.onGotoRemote(id);         break;
            }
        },

        /**
         * Show delete confirmation dialog.
         */
        onDeleteConfirmation: function (id) {
            var dialog = new Agent.Applinks.TableConfirmationDialog({
                id: id
            });

            // on confirm event, remove item from collection
            dialog.on('confirm', _.bind(function(event, id) {
                this.collection.get(id).destroy();
            }, this));

            dialog.show();
            event.preventDefault();
        },

        /**
         * Mark selected application link as primary.
         */
        onMakePrimary: function(id) {
            this.collection.get(id).makePrimary({
                success: _.bind(function() {
                    this.collection.fetch().complete(
                        _.bind(function () {
                            this.onCollectionLoad();
                            this.onCollectionUpdate();
                        }, this)
                    );
                }, this)
            });
        },

        /**
         * Call legacy code, to show existing edit dialog.
         */
        onAdvanceDialog: function(id) {
            AppLinks.UI.hideInfoBox();

            // while applinks item (JSON)
            var application = this.collection.get(id).toJSON();

            // path link url to legacy format
            application.link[0].href = AJS.contextPath() + '/rest/applinks/1.0/applicationlink/' + id;

            // mark incoming & outgoing as available
            application.hasIncoming = true;
            application.hasOutgoing = true;

            var reload = function(updated) {
                location.reload();
                return true;
            }

            AppLinks.editAppLink(application, "undefined", false, reload, reload);
        },

        /**
         * Open remote in new window.
         */
        onGotoRemote: function(id) {
            var displayUrl = this.collection.get(id).get('displayUrl');
            var $container = AJS.$('<div/>').text(displayUrl);

            window.open($container.html() + '/plugins/servlet/applinks/listApplicationLinks');
        },

        /**
         * This array will hold an array of ids that needs to be
         * @private
         */
        _pendingIds: [],
        _isRequesting: false,

        /**
         * Promote the id to the start of the queue
         * @param id
         */
        promotePendingId: function(newId, force) {
            var ids = this._pendingIds;
            var found = false;
            for (var j = ids.length; j-- > 0;) {
                if (ids[j] == newId) {
                    found = true;
                    ids.splice(j, 1);
                }
            }
            if (found || force) ids.unshift(newId);
        },

        /**
         * Fetch asynchronous all remote information.
         */
        onFetchStatus: function (newIds, force) {
            var t = this;
            var ids = t._pendingIds;

            if (newIds) {
                for(var i = newIds.length; i-- > 0;) {
                    t.promotePendingId(newIds[i], true);
                }
            }


            if ((!t._isRequesting || force) && ids.length) {
                // remove first id
                var id = ids.shift();
                var url = AJS.contextPath() + '/rest/applinks/3.0/config/remote/' + id;


                if (!force) t._isRequesting = true;
                this.collection.get(id).set({currentState: 'in-progress'});
                // get status via REST api
                AJS.$.getJSON(url, _.bind(function(data) {
                    if (data) {
                        // update remote information
                        this.collection.get(id).setRemoteValues(data);
                        this.collection.get(id).set({
                            currentState: 'responded',
                            localResponse: {}
                        });

                    }

                    // fetch other links
                    if (!force) t._isRequesting = false;
                    t.onFetchStatus();
                }, this))
                    .fail(function(response) {

                        // For all requrests that comes with an error, we will set localRequest to the model
                        /*
                         The data structure would be:
                            localResponse: {}
                                error: boolean
                                -- we might need status on rendering the table though
                                status: number -- 0, 401, 502, ...
                                messages: []
                                    $_: string -- string for displaying the tooltip
                         */
                        var messages;
                        if (response.status == 0) {
                            var messages = [AJS.I18n.getText("applinks.agent.table.timeout")];
                        } else {
                            var responseText = response.responseText || response.data;
                            try {
                                var messageJSON = AJS.$.parseJSON(responseText);
                            } catch (ex) { }
                            if (messageJSON) {
                                if (messageJSON.message) {
                                    messages = [messageJSON.message];
                                } else if (messageJSON.errors && messageJSON.errors instanceof Array) {
                                    messages = messageJSON.errors;
                                }
                            }
                            if (!messages) {
                                messages = [];
                                messages.push(AJS.I18n.getText("applinks.agent.table.local.error.generic", response.status));
                                messages.push(responseText);
                            }
                        }

                        t.collection.get(id).set({
                            currentState: 'responded',
                            localResponse: {
                                error: true,
                                status: status,
                                messages: messages
                            }});

                        if (!force) t._isRequesting = false;
                        t.onFetchStatus();
                    });
            }
        },

        untooltips: function($element) {
            // re-initialize tipsy, search only for data-title
            ($element || this.$el).find('[data-title]')
                .attr('data-title' , '');
        },

        /**
         * Initialize tipsy handler,
         * with custom className and also enable html support.
         */
            renderTooltips: function ($element) {
            // first remove any existing tipsy element
            AJS.$('body > .tipsy').remove();

            // re-initialize tipsy, search only for data-title
            ($element || this.$el).find('[data-title]')
            .tooltip({
                className: 'tipsy-left',
                title: 'data-title',
                html: true
            });
        }

    });

})(_, Backbone, agent.applinks.templates.table, AJS.$);
