/**
 * The triggering of AppLinks initialisation can be customised by setting a function on AJS.AppLinksInitialisationBinder.
 * The binder function should take a single argument which is a zero-arg function to run and should execute this function
 * when appropriate.
 */
AppLinks = AJS.$.extend(window.AppLinks || {}, {
    Event: {
        NAMESPACE: "applinks"
    }
});
AppLinks.Event = AJS.$.extend(window.AppLinks.Event, {
    PREREADY: AppLinks.Event.NAMESPACE + ".preready",
    READY: AppLinks.Event.NAMESPACE + ".ready"
});

// Is there an overridden initialisation binder?
if (AJS.AppLinksInitialisationBinder) {
    AppLinks.initialisationBinder = AJS.AppLinksInitialisationBinder;
} else {
    // The default bind if no specific binder is specified
    AppLinks.initialisationBinder = function(f) {
        AJS.toInit(f);
    }
}

AppLinks.initialisationBinder(function() {
    //$ is passed in by AJS.toInit but initilizationBinder can be over loaded by products so that $ is unsafe.
    var $ = AJS.$;
    AppLinks = $.extend(window.AppLinks || {}, {
        failure: function(data) {
            if (data.status == 401) {
                window.location.reload();
            } else {
                var message = AppLinks.parseError(data);
                var errorDivs = $('.page-error');

                if (errorDivs.length > 0) {
                    errorDivs.html(message).fadeIn('slow');
                }
                else {
                    alert("REST request failed: " + message);
                }
            }
        },
        jsonRequest: function(url, type, data, success, error) {
            if (data) {
                data = JSON.stringify(data);
            }
            $(".page-error").fadeOut('fast');
            if (!error) error = AppLinks.failure;
            return jQuery.ajax({
                url: url,
                type: type,
                data: data,
                dataType: 'json',
                contentType: "application/json; charset=utf-8",
                cache: false,
                success: success,
                error: error
            });
        },
        xmlRequest: function(url, type, data, success, error) {
            if (data) {
                data = JSON.stringify(data);
            }
            $(".page-error").fadeOut('fast');
            if (!error) error = AppLinks.failure;
            return jQuery.ajax({
                url: url,
                type: type,
                data: data,
                dataType: 'xml',
                contentType: "application/xml; charset=utf-8",
                cache: false,
                success: success,
                error: error
            });
        },
        parseError: function(errorData) {
            var error;
            try {
                error = JSON.parse(errorData.responseText);
            } catch (e) {
                if (errorData.statusText) {
                    return error = errorData.statusText;
                } else {
                    return errorData;
                }
            }
            if (error.message) {
                if ($.isArray(error.message)) {
                    return error.message.join(' ');
                }
                return error.message;
            }
            else {
                return errorData.statusText;
            }
        },
        put: function(url, data, success, error) {
            return AppLinks.jsonRequest(url, 'PUT', data, success, error);
        },
        post: function(url, data, success, error) {
            return AppLinks.jsonRequest(url, 'POST', data, success, error);
        },
        update: function(data, success, error) {
            AppLinks.put(AppLinks.self_link(data), data, success, error);
        },
        get: function(url, success, error) {
            return AppLinks.jsonRequest(url, 'GET', null, success, error);
        },
        getXml: function(url, success, error) {
            return AppLinks.xmlRequest(url, 'GET', null, success, error);
        },
        self_link: function(item) {
            for (var i = 0, _i = item.link.length; i < _i; i++) {
                var link = item.link[i];
                if (link.rel == "self") return link.href;
            }

            throw "No self-link found";
        },
        del: function(urlOrObject, success, error) {
            var url;
            if (typeof(urlOrObject) == 'string') url = urlOrObject;
            else url = AppLinks.self_link(urlOrObject);
            return AppLinks.jsonRequest(url, 'DELETE', null, success, error);
        },
        SPI: $.extend({}, {
            API_VERSION: "1.0",
            REST_RESOURCE_URL: AJS.contextPath() + "/rest/applinks/",
            BASE_URL: AJS.contextPath() + "/rest/applinks/1.0",
            /**
             * Update the API version and associated urls.
             * @param version
             */
            setApiVersion: function(version){
                AppLinks.SPI.API_VERSION = version;
                AppLinks.SPI.setBaseUrl(AppLinks.SPI.REST_RESOURCE_URL + AppLinks.SPI.API_VERSION);
            },
            setBaseUrl: function(url){
                AppLinks.SPI.BASE_URL = url;
            },
            /**
             * Build a base URL for rest calls using the specified baseUrl.
             * @param baseUrl
             * @returns {string}
             */
            getRemoteRestBaseUrl: function(baseUrl) {
                return baseUrl + "/rest/applinks/" + AppLinks.SPI.API_VERSION;
            },
            /**
             * Build a base URL for plugin servlet calls using the specified baseUrl.
             * @param baseUrl
             * @returns {string}
             */
            getRemotePluginServletBaseUrl: function(baseUrl) {
                return baseUrl + "/plugins/servlet";
            },
            getAllLinks: function(success, failure) {
                var url = AppLinks.SPI.BASE_URL + "/applicationlink";
                return AppLinks.get(url, success, failure);
            },
            getAllLinksWithAuthInfo: function(success, failure) {
                var url = AppLinks.SPI.BASE_URL + "/listApplicationlinks";
                return AppLinks.get(url, success, failure);
            },
            getApplicationLinkState: function(id, success, failure) {
                var url = AppLinks.SPI.BASE_URL + "/listApplicationlinkstates/id/" + id;
                return AppLinks.get(url, success, failure);
            },
            getLinksOfType: function(typeId, success, failure) {
                var url = AppLinks.SPI.BASE_URL + "/applicationlink/type/" + typeId;
                return AppLinks.get(url, success, failure);
            },
            tryToFetchManifest: function(url, success, failure) {
                var restUrl = AppLinks.SPI.BASE_URL + '/applicationlinkForm/manifest.json?url=' + encodeURIComponent(url);
                return AppLinks.get(restUrl, success, failure);
            },
            getManifestFor: function(id, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/manifest/' + id + ".json";
                return AppLinks.get(url, success, failure);
            },
            getLocalManifest: function(success, failure){
                var url = AppLinks.SPI.BASE_URL + '/manifest.json';
                return AppLinks.get(url, success, failure);
            },
            /**
             * Attempt to get the Manifest of the remote application, via a direct REST call.
             * Requires CORS enabled on the REST resource.
             * @param url
             * @param success
             * @param failure
             * @returns {*}
             */
            getRemoteManifest: function(remoteBaseUrl, success, failure){
                var remoteManifestUrl = AppLinks.SPI.getRemoteRestBaseUrl(remoteBaseUrl) + '/manifest.json';
                return AppLinks.get(remoteManifestUrl, success, failure);
            },
            /**
             * Attempt to get the OAuth Consumer Info of the remote application, via a direct call.
             * Requires CORS enabled on the REST resource.
             * @param url
             * @param success
             * @param failure
             * @returns {*}
             */
            getRemoteOAuthConsumerInfo: function(remoteBaseUrl, success, failure){
                var remoteManifestUrl = AppLinks.SPI.getRemotePluginServletBaseUrl(remoteBaseUrl) + '/oauth/consumer-info';
                return AppLinks.getXml(remoteManifestUrl, success, failure);
            },
            createStaticUrlAppLink: function(applicationType, success, failure) {
                var restUrl = AppLinks.SPI.BASE_URL + '/applicationlinkForm/createStaticUrlAppLink?typeId=' + applicationType;
                return AppLinks.post(restUrl, null, success, failure);
            },
            createLink: function(applicationLink, username, password, createTwoWayLink, customRpcUrl, rpcUrl, configFormValues, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlinkForm/createAppLink';
                var data = {
                    applicationLink: applicationLink,
                    username: username,
                    password: password,
                    createTwoWayLink: createTwoWayLink,
                    customRpcURL: customRpcUrl,
                    rpcUrl: rpcUrl,
                    configFormValues: configFormValues
                };
                return AppLinks.post(url, data, success, failure);
            },
            createLinkWithOrphanedTrust : function(applicationLink, username, password, createTwoWayLink, customRpcUrl, rpcUrl, configFormValues, orphanedTrust, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlinkForm/createAppLink';
                var data = {
                    applicationLink: applicationLink,
                    username: username,
                    password: password,
                    createTwoWayLink: createTwoWayLink,
                    customRpcURL: customRpcUrl,
                    rpcUrl: rpcUrl,
                    configFormValues: configFormValues,
                    orphanedTrust: orphanedTrust
                };
                return AppLinks.post(url, data, success, failure);
            },
            verifyTwoWayLinkDetails : function (remoteUrl, rpcUrl, username, password, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlinkForm/details';
                var data = {
                    username: username,
                    password: password,
                    remoteUrl: remoteUrl,
                    rpcUrl: rpcUrl
                }
                return AppLinks.post(url, data, success, failure);
            },
            getApplicationLinkInfo: function (appId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/applicationlinkInfo/id/" + appId;
                return AppLinks.get(url, success, error);
            },
            deleteLink: function(applicationLink, reciprocate, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/applicationlink/" + applicationLink.id;
                if (reciprocate) url += "?reciprocate=true";
                return AppLinks.del(url, success, error);
            },
            makePrimary: function(applicationLink, success) {
                var url = AppLinks.SPI.BASE_URL + "/applicationlink/primary/" + applicationLink.id;
                return AppLinks.post(url, null, success);
            },
            relocate: function(applicationLink, newUrl, suppressWarnings, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/relocateApplicationlink/" + applicationLink.id + "?newUrl=" + encodeURIComponent(newUrl) +
                        "&nowarning=" + (suppressWarnings ? "true" : "false");
                return AppLinks.post(url, null, success, error);
            },
            legacyUpgrade: function(applicationLink, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/upgrade/legacy/" + applicationLink.id;
                return AppLinks.post(url, null, success, error);
            },
            ualUpgrade: function(applicationLink, body, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/upgrade/ual/" + applicationLink.id;
                return AppLinks.post(url, body, success, error);
            },
            getEntityTypesForApplicationType: function(applicationType, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/type/entity/" + applicationType;
                return AppLinks.get(url, success, error);
            },
            getLocalEntitiesWithLinksToApplication: function(applicationLinkId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/localEntitiesWithLinksTo/" + applicationLinkId + ".json";
                return AppLinks.get(url, success, error);
            },
            getEntityLinksForApplication: function(applicationLinkId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entities/" + applicationLinkId + ".json";
                AppLinks.get(url, success, error);
            },
            getEntityLinksForApplicationUsingAnonymousAccess: function(applicationLinkId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entities/anonymous/" + applicationLinkId + ".json";
                return AppLinks.get(url, success, error);
            },
            createNonUalEntityLink: function(localType, localKey, applicationId, remoteTypeId, remoteKey, name, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/" + localType + "/" + localKey + "?reciprocate=false";
                var data = {
                    applicationId: applicationId,
                    typeId: remoteTypeId,
                    key: remoteKey,
                    name: name,
                    isPrimary: false
                };
                return AppLinks.put(url, data, success, error);
            },
            createEntityLink: function(localType, localKey, entity, reciprocate, success, failure) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/" + localType + "/" + localKey + "?reciprocate=";
                url += (reciprocate ? "true" : "false");
                return AppLinks.put(url, entity, success, failure);
            },
            getConfiguredEntityLinks: function(localType, localKey, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/primaryLinks/" + localType + "/" + localKey + ".json";
                return AppLinks.get(url, success, error);
            },
            deleteEntityLink: function(localTypeId, localKey, entity, reciprocate, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/" + localTypeId + "/" + localKey + "?typeId=" + entity.typeId + "&key=" + entity.key + "&applicationId=" + entity.applicationId + "&reciprocate=" + reciprocate;
                return AppLinks.del(url, success, error);
            },
            makePrimaryEntityLink: function(localTypeID, localKey, entity, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/entitylink/primary/" + localTypeID + "/" + localKey + "?typeId=" + entity.typeId + "&key=" + entity.key + "&applicationId=" + entity.applicationId;
                return AppLinks.post(url, null, success, error);
            },
            canDeleteAppLink: function(applicationId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/permission/reciprocate-application-delete/" + applicationId;
                return AppLinks.get(url, success, error);
            },
            canDeleteEntityLink: function(localTypeId, localKey, entity, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/permission/reciprocate-entity-delete/" + entity.applicationId + "/" + localTypeId + "/" + localKey + "/" + entity.typeId + "/" + entity.key;
                return AppLinks.get(url, success, error);
            },
            canCreateReciprocateEntityLink: function(applicationId, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/permission/reciprocate-entity-create/" + applicationId;
                return AppLinks.get(url, success, error);
            },
            processPermissionCode: function(settings) {
                var config = {
                    noPermission: function() {},
                    missing: function() {},
                    credentialsRequired: function(authUrl) {},
                    authenticationFailed: function(authUrl) {},
                    noAuthentication: function(authUrl) {},
                    noAuthenticationConfigured: function() {},
                    noConnection: function() {},
                    allowed: function() {},
                    unrecognisedCode: function(code) {},
                    updateView: function(message, icon, button) {}
                };

                if (!settings) settings = {};

                settings = $.extend(config, settings);

                return function(data) {
                    var code = data.code;
                    if (code == "NO_PERMISSION") {
                        settings.noPermission();
                    } else if (code == "MISSING") {
                        settings.missing();
                    } else if (code == "CREDENTIALS_REQUIRED") {
                        settings.credentialsRequired(data.url);
                    } else if (code == "AUTHENTICATION_FAILED") {
                        settings.authenticationFailed(data.url);
                    } else if (code == "NO_AUTHENTICATION") {
                        settings.noAuthentication(data.url);
                    } else if (code == "NO_AUTHENTICATION_CONFIGURED") {
                        settings.noAuthenticationConfigured();
                    } else if (code == "NO_CONNECTION") {
                        settings.noConnection();
                    } else if (code == "ALLOWED") {
                        settings.allowed();
                    } else {
                        settings.unrecognisedCode(data.code);
                    }
                };
            },
            addAuthenticationTrigger: function(target, authUrl, callbacks) {
                if (!callbacks) {
                    callbacks = {};
                }

                if (typeof callbacks.onSuccess == "undefined") {
                    callbacks.onSuccess = function() {
                        location.reload();
                    }
                }

                if (typeof callbacks.onFailure == "undefined") {
                    callbacks.onFailure = function() {
                        return true;
                    }
                }
                //Unbind previous click listener, otherwise we might end up opening multiple windows.
                $(target).unbind('click');
                $(target).click(function() {
                    if (callbacks.before) {
                        callbacks.before();
                    }
                    AppLinks.authenticateRemoteCredentials(authUrl, callbacks.onSuccess, callbacks.onFailure);
                });
            },
            deleteOrphanedTrust: function(id, type, success, error) {
                var url = AppLinks.SPI.BASE_URL + "/orphaned-trust/" + type + "/" + id;
                return AppLinks.del(url, success, error);
            },
            getOrphanedTrust: function(success, error) {
                var url = AppLinks.SPI.BASE_URL + "/orphaned-trust/";
                return AppLinks.get(url, success, error);
            },
            showCreateEntityLinkSuggestion: function() {
                return true;
            },
            getApplicationLink: function(id, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlink/' + id;
                return AppLinks.get(url, success, failure);
            },
            createApplicationLink: function(id, name, rpcUrl, displayUrl, typeId, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlink';
                var data = {
                    id: id,
                    name: name,
                    rpcUrl: rpcUrl,
                    displayUrl: displayUrl,
                    typeId: typeId
                };
                return AppLinks.put(url, data, success, failure);
            },
            createConsumer: function(id, key, name, description, sharedSecret, publicKey, twoLOAllowed, executingTwoLOUser, twoLOImpersonationAllowed, outgoing, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlink/' + id + '/authentication/consumer';
                var data = {
                    key: key,
                    name: name,
                    description: description,
                    sharedSecret: sharedSecret,
                    publicKey: publicKey,
                    outgoing: outgoing,
                    twoLOAllowed: twoLOAllowed,
                    executingTwoLOUser: executingTwoLOUser,
                    twoLOImpersonationAllowed: twoLOImpersonationAllowed
                };
                return AppLinks.put(url, data, success, failure);
            },
            createConsumerAutoConfigure: function(id, twoLOAllowed, executingTwoLOUser, twoLOImpersonationAllowed, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlink/' + id + '/authentication/consumer?autoConfigure=true';
                var data = {
                    twoLOAllowed: twoLOAllowed,
                    executingTwoLOUser: executingTwoLOUser,
                    twoLOImpersonationAllowed: twoLOImpersonationAllowed
                };
                return AppLinks.put(url, data, success, failure);
            },
            registerProvider: function(id, provider, config, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/applicationlink/' + id + '/authentication/provider';
                var data = {
                    config : config,
                    provider : provider
                };
                return AppLinks.put(url, data, success, failure);
            },
            enableFeature: function(featureName, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/features/' + featureName;
                return AppLinks.put(url, {}, success, failure);
            },
            disableFeature: function(featureName, success, failure) {
                var url = AppLinks.SPI.BASE_URL + '/features/' + featureName;
                return AppLinks.del(url, success, failure);
            }
        }, (window.AppLinks && window.AppLinks.SPI) || {})
    });

    var i18nRootKey = 'applinks';

    AppLinks.UI = {
        showInfoBox: function(message) {
            $('.aui-message.success').remove();
            AppLinks.UI.createMessage('success', message, 'page-info');
        },
        hideInfoBox: function() {
            $('.aui-message.success').remove();
        },
        showErrorBox: function(message) {
            AppLinks.UI.createMessage('error', message, 'page-error');
        },
        hideErrorBox: function() {
            $('.aui-message.error').remove();
        },
        showWarningBox: function(messages) {
            if ($.isArray(messages) && messages.length > 0) {
                var ulEl = $("<ul></ul>");
                $(messages).each(function(index) {
                    ulEl.append($("<li/>", {
                        text: messages[index]
                    }));
                });
                var messageEl = $('<div class="page-warning"></div>').append(ulEl);
                AppLinks.UI.createMessage('warning', messageEl.html(), 'page-warning');
            } else {
                AppLinks.UI.createMessage('warning', messages, 'page-warning');
            }
        },
        hideWarningBox: function() {
            $('.aui-message.warning').remove();
        },
        shortenString: function(message, maxLength) {
            if (message.length  > maxLength) {
               message = message.substring(0, maxLength) + "...";
            }
            return message;
        },
        createMessage: function(type, message, cssClass) {
            var messageEl = $('<div class="' + cssClass + '">');
            messageEl.html(message);
            AJS.messages[type](".applinks-message-bar", {
                title: "",
                body: messageEl.wrap('<div></div>').parent().html(),
                closeable: true
            });
        },
        displayValidationErrorMessages: function (errorClass, rootEl, messages) {
            if ($.isArray(messages)) {
                $(messages).each(function(i,v) {
                   var d = $('<div class="error applinks-error">');
                   d.text(v);
                   $(rootEl).find("." + errorClass).append(d);
                });
            } else if(typeof messages != 'undefined'){
                var d = $('<div class="error applinks-error">');
                d.text(messages.toString());
                $(rootEl).find("." + errorClass).append(d);
            }
        },
        displayValidationError: function(errorClass, rootEl, errorFn) {
            return function(xhr) {
                if (xhr.status == 401) {
                    window.location.reload();
                    return;
                }

                $('.applinks-error').remove();
                $('.loading').remove();
                var respJSON = xhr.responseText;
                var respObj = $.parseJSON(respJSON);
                var messages = respObj.message;
                if (typeof respObj.fields == "undefined") {
                    AppLinks.UI.displayValidationErrorMessages(errorClass, rootEl, messages);
                } else {
                    var fields = respObj.fields;
                    $(fields).each(function(index) {
                        var d = $('<div class="error applinks-error" id="' + fields[index] + '-error">');
                        d.text(messages[index]);
                        if ($(rootEl).find('.' + fields[index]).length > 0) {
                          d.insertAfter($(rootEl).find('.' + fields[index]));
                        } else {
                          d.insertAfter($(rootEl).find('.' + errorClass).append(d));
                        }
                    });
                }
                $(rootEl).find('.' + errorClass).addClass("fully-populated-errors");
                if (errorFn) {
                 errorFn();
                }
            }
        },
        addProtocolToURL : function(url) {
            var newUrl = $.trim(url);
            var tempURL = newUrl.toLowerCase();
            var hasProtocol = false;
            if (tempURL.length >= 7) {
                if (tempURL.substring(0,7).indexOf('http') != -1) {
                    hasProtocol = true;
                }
            }
            //default protocol is http
            if (!hasProtocol) {
                newUrl = 'http://' + newUrl;
            }
            return newUrl;
        },
        /**
         * Similar to the standard Javascript join() method, but nicer in that
         * it uses a different delimiter for the last node (by default "and"),
         * so that:
         * {code}
         * "1, 2 and 3" == prettyJoin(['1', '2', '3'], function(value) {return value;});
         * {code}
         *
         * @param inputArray
         * @param resolveFn
         * @param finalDelimiter
         */
        prettyJoin : function(inputArray, resolveFn, finalDelimiter) {
            if (!finalDelimiter) {
                finalDelimiter = AppLinks.I18n.getText('applinks.and');
            }
            var maxLength = inputArray.length;
            var message = "";
            $.each(inputArray, function(index, value) {
                if (index == (maxLength - 1) && maxLength > 1) {
                  message += " " + finalDelimiter + "  " + resolveFn(value);
                } else {
                  message += resolveFn(value);
                  if (index + 2 < maxLength) {
                      message += ", ";
                  }
                }
            });
            return message;
        },
        showLoadingIcon: function(element) {
            $('<span class="loading">&nbsp;</span>').insertAfter(element);
        },
        hideLoadingIcon: function(element) {
            $(element).next('.loading').remove();
        },
        findUrl: function(text) {
            var url = undefined;
            var lcText = text.toLowerCase();
            var startOfUrl = lcText.indexOf('http:');
            if (startOfUrl == -1) {
                startOfUrl = lcText.indexOf('https:');
            }
            if (startOfUrl > -1) {
                var endOfUrl = lcText.indexOf(' ', startOfUrl);
                if (endOfUrl == -1) {
                    endOfUrl = lcText.length;
                }
                url = text.substring(startOfUrl, endOfUrl); // use _case-sensitive_ version to retrieve the actual URL
            }
            return url;
        },
        findApplicationType : function(id) {
            id = id.toLowerCase();
            if (id.indexOf("jira") != -1) {
                return "jira";
            } else if (id.indexOf("fisheye") != -1) {
                return "fecru";
            } else if (id.indexOf("confluence") != -1) {
                return "confluence";
            } else if (id.indexOf("refapp") != -1) {
                return "refapp";
            } else {
                return undefined;
            }
        },
        escapeSelector: function(selector) {
            // based on http://samuelsjoberg.com/archive/2009/09/escape-jquery-selectors
            return selector.replace(/([#;&,\.\+\*\~':"\!\^$\[\]\(\)=>\|])/g, "\\$1");
        },
        sanitiseHTML: function(input) {
            var replacements = {
                "<": "&lt;",
                '"': "&quot;",
                "&": "&amp;"
            };
            return input.replace(/[<"&]/g, function(match) {
                return replacements[match];
            });
        },
        refreshOrphanedTrust: function() {
            // post dialog -- check whether we need to remove any orphaned-trust entries
            var updateOrphanedTrust = function(data) {
                $("tr.orphaned-trust-row").each(function() {
                    var $this = $(this);
                    var id = $this.attr("data-id");
                    var type = $this.attr("data-type");
                    var stillExists = false;
                    for (var i = 0; i < data.orphanedTrust.length; i++) {
                        var ot = data.orphanedTrust[i];
                        if (id == ot.id && type == ot.type) {
                            stillExists = true;
                            break;
                        }
                    }
                    if (!stillExists) {
                        $this.remove();
                        if (data.orphanedTrust.length == 0) {
                            // we just removed the last orphaned trust cert, hide warning!
                            $(".orphaned-trust-warning").hide();
                        }
                    }
                });
            };

            AppLinks.SPI.getOrphanedTrust(updateOrphanedTrust);
        },
        removeCssClass: function(element, prefix) {
            $(element).removeClass( function(index, className) {
                   var classes = className.split(' ');
                   var classToRemove = "";
                   $.each(classes, function(index, value) {
                       if (value.indexOf(prefix) != -1) {
                           classToRemove = value;
                       }
                   });
                   return classToRemove;
            } );
        }
    };

    /**
     * Add jQuery event system to AppLinks.UI namespace.
     */
    (function(){
        var eventBus = $({});
        $.each(['bind', 'unbind', 'trigger'], function(i, current){
            AppLinks.UI[current] = function(){
                return eventBus[current].apply(eventBus, arguments);
            }
        });
    })();

    AppLinks.I18n = {
        // very simple i18n param interpolation, doesn't attempt to respect escaping
        //Deferrs to AJS.format for for legacy reason the array conversion needs to stick around.
        interpolate: function(text, params) {
            if (params) {
                if (!$.isArray(params)) {
                    // single arg
                    params = [new String(params)];
                }
                params.unshift(text);
                text = AJS.format.apply(AJS, params);
            }
            return text;
        },
        getTextWithPrefix: function(key, params) {
            return AppLinks.I18n.interpolate(appLinksI18n.entries[i18nRootKey + "." + key], params);
        },
        getText: function(key, params) {
            return AppLinks.I18n.interpolate(AppLinks.I18n.resolveValue(key), params);
        },
        getApplicationTypeName: function(typeId) {
            return appLinksI18n.entries["applinks.application.type." + typeId];
        },
        getEntityTypeName: function(typeId) {
            return appLinksI18n.entries["applinks.entity.type." + typeId];
        },
        getPluralizedEntityTypeName: function(typeId) {
            return appLinksI18n.entries["applinks.entity.type.plural." + typeId];
        },
        getAuthenticationTypeName: function(type) {
            return appLinksI18n.entries["applinks.auth.provider." + type];
        },
        resolveValue: function(key) {
            var value = appLinksI18n.entries[key];
            return typeof value == "undefined" ? key : value;
        }
    };

    AppLinks.Docs = {
        /**
         * NOTE: this is a dynamically generated version of the link build in _help_link.vm, any update here should be
         * applied there.
         * @method createDocLink
         * @param pageKey a key that maps to a page in ual-help-paths.properties
         * @param sectionKey (Optional) a key that maps to an anchor section id in ual-help-paths.properties
         * @return an html &lt;a&gt; element targeting the specified page & section
         */
        createDocLink: function(pageKey, sectionKey, css) {
            if (!css) {
                css = "";
            } else {
                css = " " + css;
            }
            return $("<a/>", {
                "class": "ual-help-link" + css,
                href: AppLinks.Docs.getDocHref(pageKey, sectionKey),
                target: "_blank",
                text: AppLinks.I18n.getText("applinks.help"),
                title: AppLinks.I18n.getText("applinks.help")
            });
        },
        /**
         * @method getDocHref
         * @param pageKey a key that maps to a page in ual-help-paths.properties
         * @param sectionKey (Optional) a key that maps to an anchor section id in ual-help-paths.properties
         * @return the url of the given page and section (if specified)
         */
        getDocHref: function(pageKey, sectionKey) {
            var link = AppLinks.Docs.resolveValue('applinks.docs.root') + '/' + AppLinks.Docs.resolveValue(pageKey);
            if (sectionKey) {
                link += '#' + AppLinks.Docs.resolveValue(sectionKey);
            }
            return link;
        },
        resolveValue: function(key) {
            var value = applinksDocs.entries[key];
            return typeof value == "undefined" ? key : value;
        }
    };

    $(document).trigger(AppLinks.Event.PREREADY);
    $(document).trigger(AppLinks.Event.READY);
});
