package com.atlassian.application.host.plugin;

import io.atlassian.fugue.Option;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;

/**
 * Some utility methods related to parsing and processing {@code DateTime} instances.
 *
 * @since v1.0
 */
final class PluginDateTimeUtils
{
    private static final DateTimeFormatter BUILD_DATE_FORMAT
            = new DateTimeFormatterBuilder()
            .appendPattern("yyyy-MM-dd'T'HH:mm:ss")
            .appendOffset("+HHMM", "Z")
            .toFormatter();

    /**
     * Cannot create instances.
     */
    private PluginDateTimeUtils()
    {
    }

    /**
     * Parse the passed string a plugin date and time. {@link io.atlassian.fugue.Option#none()} is returned
     * if the passed string is not a valid plugin date and time.
     *
     * @param date the string to parse.
     *
     * @return the parsed {@link java.time.ZonedDateTime} or {@link io.atlassian.fugue.Option#none()} if the string
     * does not parse.
     */
    static Option<ZonedDateTime> fromPluginString(final String date)
    {
        try
        {
            return Option.some(ZonedDateTime.parse(date, BUILD_DATE_FORMAT));
        }
        catch (final DateTimeParseException e)
        {
            return Option.none();
        }
    }

    /**
     * Turn the passed {@link ZonedDateTime} into a String.
     *
     * @return the passed {@code DateTime} as a string.
     */
    static String toPluginString(final ZonedDateTime date)
    {
        return date.format(BUILD_DATE_FORMAT);
    }
}
