package com.atlassian.application.host.plugin;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.osgi.factory.OsgiPlugin;
import com.google.common.base.Function;
import io.atlassian.fugue.Option;
import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.ZonedDateTime;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import static io.atlassian.fugue.Option.none;
import static io.atlassian.fugue.Option.option;

/**
 * Parses the AMPS build date embedded in the {@code Atlassian-Build-Date} OSGi manifest header.
 *
 * @since v1.0
 */
@Immutable
@ThreadSafe
final class AmpsBuildDateParser implements Function<Plugin, Option<ZonedDateTime>>
{
    private static final Logger log = LoggerFactory.getLogger(AmpsBuildDateParser.class);

    private static final String BUILD_DATE_ATTRIBUTE = "Atlassian-Build-Date";

    /**
     * Creates a new date parser.
     */
    AmpsBuildDateParser()
    {
    }

    @Override
    public Option<ZonedDateTime> apply(@Nullable final Plugin plugin)
    {
        if (plugin == null)
        {
            return none();
        }
        else
        {
            for (Bundle bundle : getPluginBundle(plugin))
            {
                final Object value = bundle.getHeaders().get(BUILD_DATE_ATTRIBUTE);
                if (value != null)
                {
                    final Option<ZonedDateTime> buildDate = PluginDateTimeUtils.fromPluginString(value.toString());
                    if (buildDate.isEmpty())
                    {
                        log.debug("Plugin with key \"{}\" has invalid Atlassian-Build-Date of \"{}\".",
                                plugin.getKey(), value);
                    }
                    return buildDate;
                }
            }
            return none();
        }
    }

    private static Option<Bundle> getPluginBundle(final Plugin plugin)
    {
        if (plugin instanceof OsgiPlugin)
        {
            return option(((OsgiPlugin) plugin).getBundle());
        }
        return none();
    }
}
