package com.atlassian.application.host.plugin;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.application.api.ApplicationPlugin;
import com.atlassian.plugin.Plugin;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import io.atlassian.fugue.Option;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.lang.ref.WeakReference;
import java.net.URI;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import javax.annotation.Nullable;

import static com.atlassian.application.api.ApplicationPlugin.PluginType;
import static com.atlassian.application.api.ApplicationPlugin.PluginType.APPLICATION;
import static com.atlassian.application.api.ApplicationPlugin.PluginType.PRIMARY;
import static com.atlassian.application.api.ApplicationPlugin.PluginType.UTILITY;
import static com.atlassian.application.host.util.Arguments.checkArgumentNotBlank;
import static com.atlassian.application.host.util.Arguments.checkArgumentNotNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * Builder for {@link PluginApplicationMetaData}.
 *
 * @since v1.0
 */
class PluginApplicationMetaDataBuilder
{
    private ApplicationKey key;
    private String name;
    private Option<URI> configURI = Option.none();
    private Option<URI> postInstallURI = Option.none();
    private Option<URI> postUpdateURI = Option.none();
    private Option<URI> productHelpServerSpaceURI = Option.none();
    private Option<URI> productHelpCloudSpaceURI = Option.none();
    private String definitionModuleKey;
    private ApplicationPlugin primaryPlugin;
    private Iterable<ApplicationPlugin> applicationPlugins = Collections.emptyList();
    private Iterable<ApplicationPlugin> utilityPlugins = Collections.emptyList();
    private String descriptionKey;
    private String userCountKey;
    private Plugin plugin;
    private String defaultGroup;

    PluginApplicationMetaDataBuilder()
    {
    }

    PluginApplicationMetaDataBuilder key(final ApplicationKey key)
    {
        this.key = checkArgumentNotNull(key, "key is null");
        return this;
    }

    PluginApplicationMetaDataBuilder name(final String name)
    {
        this.name = checkArgumentNotBlank(name, "name is null");
        return this;
    }

    PluginApplicationMetaDataBuilder configURI(final Option<URI> uri)
    {
        this.configURI = checkArgumentNotNull(uri, "uri is null");
        return this;
    }

    PluginApplicationMetaDataBuilder postInstallURI(final Option<URI> uri)
    {
        this.postInstallURI = checkArgumentNotNull(uri, "uri is null");
        return this;
    }

    PluginApplicationMetaDataBuilder postUpdateURI(final Option<URI> uri)
    {
        this.postUpdateURI = checkArgumentNotNull(uri, "uri is null");
        return this;
    }

    PluginApplicationMetaDataBuilder productHelpServerSpaceURI(final Option<URI> uri)
    {
        this.productHelpServerSpaceURI = checkArgumentNotNull(uri, "help server space uri is null");
        return this;
    }

    PluginApplicationMetaDataBuilder productHelpCloudSpaceURI(final Option<URI> uri)
    {
        this.productHelpCloudSpaceURI = checkArgumentNotNull(uri, "help cloud space uri is null");
        return this;
    }

    PluginApplicationMetaDataBuilder definitionModuleKey(final String definitionModuleKey)
    {
        this.definitionModuleKey = checkArgumentNotBlank(definitionModuleKey, "definitionModuleKey");
        return this;
    }

    PluginApplicationMetaDataBuilder primaryPlugin(final Plugin plugin)
    {
        this.plugin = checkArgumentNotNull(plugin, "plugin");
        this.primaryPlugin = new DefaultApplicationPlugin(plugin.getKey(), PRIMARY);
        return this;
    }

    PluginApplicationMetaDataBuilder applicationPlugins(final Iterable<String> keys)
    {
        this.applicationPlugins = createPlugins(keys, APPLICATION);
        return this;
    }

    PluginApplicationMetaDataBuilder utilityPlugins(final Iterable<String> keys)
    {
        this.utilityPlugins = createPlugins(keys, UTILITY);
        return this;
    }

    PluginApplicationMetaDataBuilder descriptionKey(final String key)
    {
        this.descriptionKey = checkArgumentNotBlank(key, "key");
        return this;
    }

    PluginApplicationMetaDataBuilder userCountKey(final String key)
    {
        this.userCountKey = checkArgumentNotBlank(key, "key");
        return this;
    }

    PluginApplicationMetaDataBuilder defaultGroup(final String defaultGroup)
    {
        this.defaultGroup = checkArgumentNotBlank(defaultGroup, "defaultGroup is null");
        return this;
    }

    private static List<ApplicationPlugin> createPlugins(final Iterable<String> keys, final PluginType pluginType)
    {
        checkArgumentNotNull(keys, "keys");

        final List<ApplicationPlugin> metaData = Lists.newArrayList();
        for (@Nullable String key : Sets.newHashSet(keys))
        {
            if (key == null)
            {
                throw new IllegalArgumentException("keys contains a null value.");
            }

            metaData.add(new DefaultApplicationPlugin(key, pluginType));
        }
        return metaData;
    }

    PluginApplicationMetaData build()
    {
        checkState(key != null, "key is null");
        checkState(name != null, "productName is null");
        checkState(definitionModuleKey != null, "definitionModuleKey is null");
        checkState(primaryPlugin != null, "primaryPlugin is null");
        checkState(descriptionKey != null, "descriptionKey is null");
        checkState(userCountKey != null, "userCountKey is null");
        checkState(defaultGroup != null, "defaultGroup is null");

        return new DefaultPluginApplicationMetaData(this);
    }

    private static final class DefaultPluginApplicationMetaData implements PluginApplicationMetaData
    {
        private final ApplicationKey key;
        private ZonedDateTime buildDate;
        private String version;
        private final String name;
        private final Option<URI> configURI;
        private final Option<URI> postUpdateURI;
        private final Option<URI> postInstallURI;
        private final Option<URI> productHelpServerSpaceURI;
        private final Option<URI> productHelpCloudSpaceURI;
        private final String definitionModuleKey;
        private final ApplicationPlugin primaryPlugin;
        private final List<ApplicationPlugin> applicationPlugins;
        private final List<ApplicationPlugin> utilityPlugins;
        private final String descriptionKey;
        private final String userCountKey;
        private final WeakReference<Plugin> plugin;
        private final String defaultGroup;

        private DefaultPluginApplicationMetaData(final PluginApplicationMetaDataBuilder builder)
        {
            this.key = builder.key;
            this.name = builder.name;
            this.configURI = builder.configURI;
            this.postUpdateURI = builder.postUpdateURI;
            this.postInstallURI = builder.postInstallURI;
            this.definitionModuleKey = builder.definitionModuleKey;
            this.primaryPlugin = builder.primaryPlugin;
            this.applicationPlugins = ImmutableList.copyOf(builder.applicationPlugins);
            this.utilityPlugins = ImmutableList.copyOf(builder.utilityPlugins);
            this.descriptionKey = builder.descriptionKey;
            this.userCountKey = builder.userCountKey;
            this.defaultGroup = builder.defaultGroup;
            this.productHelpServerSpaceURI = builder.productHelpServerSpaceURI;
            this.productHelpCloudSpaceURI = builder.productHelpCloudSpaceURI;

            //We don't want to keep a strong reference to the plugin.
            this.plugin = new WeakReference<Plugin>(builder.plugin);
        }

        @Override
        public String getDefinitionModuleKey()
        {
            return this.definitionModuleKey;
        }

        @Override
        public Iterable<ApplicationPlugin> getPlugins()
        {
            final List<Iterable<ApplicationPlugin>> plugins = Lists.newArrayList();
            plugins.add(Collections.singleton(getPrimaryPlugin()));
            plugins.add(getApplicationPlugins());
            plugins.add(getUtilityPlugins());

            return Iterables.concat(plugins);
        }

        @Override
        public ApplicationPlugin getPrimaryPlugin()
        {
            return primaryPlugin;
        }

        @Override
        public Iterable<ApplicationPlugin> getApplicationPlugins()
        {
            return applicationPlugins;
        }

        @Override
        public Iterable<ApplicationPlugin> getUtilityPlugins()
        {
            return utilityPlugins;
        }

        @Override
        public ApplicationKey getKey()
        {
            return key;
        }

        @Override
        public String getName()
        {
            return name;
        }

        @Override
        public String getVersion()
        {
            //We have to do this lazily because the version might not be available when the module descriptor is parsed
            if (version == null)
            {
                final Option<Plugin> op = Option.option(plugin.get());
                if (op.isDefined())
                {
                    for (Plugin p : op)
                    {
                        version = p.getPluginInformation().getVersion();
                    }
                }
                else
                {
                    throw new IllegalStateException("Plugin version is currently unknown");
                }
            }
            return version;
        }

        @Override
        public String getDescriptionKey()
        {
            return descriptionKey;
        }

        @Override
        public String getUserCountKey()
        {
            return userCountKey;
        }

        @Override
        public Option<URI> getConfigurationURI()
        {
            return configURI;
        }

        @Override
        public Option<URI> getPostInstallURI()
        {
            return postInstallURI;
        }

        @Override
        public Option<URI> getPostUpdateURI()
        {
            return postUpdateURI;
        }

        @Override
        public String getDefaultGroup()
        {
            return defaultGroup;
        }

        @Override
        public Option<URI> getProductHelpServerSpaceURI()
        {
            return productHelpServerSpaceURI;
        }

        @Override
        public Option<URI> getProductHelpCloudSpaceURI()
        {
            return productHelpCloudSpaceURI;
        }

        @Override
        public ZonedDateTime buildZonedDate()
        {
            //We have to do this lazily because you can't read the OSGi header while the plugin is being inited.
            if (buildDate == null)
            {
                final Plugin plugin = this.plugin.get();
                if (plugin == null)
                {
                    throw new IllegalStateException("Plugin '" + getPrimaryPlugin().getPluginKey()
                            + "' no longer exists.");
                }
                buildDate = new AmpsBuildDateParser().apply(plugin)
                        .getOrElse(ZonedDateTime.ofInstant(Instant.EPOCH, ZoneId.systemDefault()));
            }
            return buildDate;
        }

        @Override
        public String toString()
        {
            return new ToStringBuilder(this)
                    .append("key", key)
                    .append("buildDate", buildDate)
                    .append("productName", name)
                    .append("configURI", configURI)
                    .append("postUpdateURI", postUpdateURI)
                    .append("postInstallURI", postInstallURI)
                    .append("productHelpServerSpaceURI", productHelpServerSpaceURI)
                    .append("productHelpCloudSpaceURI", productHelpCloudSpaceURI)
                    .append("definitionModuleKey", definitionModuleKey)
                    .append("primaryPlugin", primaryPlugin)
                    .append("applicationPlugins", applicationPlugins)
                    .append("utilityPlugins", utilityPlugins)
                    .append("descriptionKey", descriptionKey)
                    .append("userCountKey", userCountKey)
                    .append("defaultGroup", defaultGroup)
                    .toString();
        }

        //This only looks at the application key on purpose.
        @Override
        public boolean equals(final Object o)
        {
            if (this == o)
            {
                return true;
            }

            if (o == null || getClass() != o.getClass())
            {
                return false;
            }

            final DefaultPluginApplicationMetaData that = (DefaultPluginApplicationMetaData) o;
            return key.equals(that.key);
        }

        //This only looks at the application key on purpose.
        @Override
        public int hashCode()
        {
            return key.hashCode();
        }
    }
}
