package com.atlassian.application.host;

import com.atlassian.application.api.ApplicationAccess;
import com.atlassian.application.api.ApplicationKey;
import com.atlassian.application.host.util.Arguments;
import com.atlassian.sal.api.user.UserKey;
import io.atlassian.fugue.Option;

import java.net.URI;
import javax.annotation.Nullable;

/**
 * An implementation of {@link com.atlassian.application.api.ApplicationAccess} when the application is not
 * licensed. In this case:
 * <ul>
 *     <li>{@link #getMaximumUserCount()} returns 0.</li>
 *     <li>{@link #getActiveUserCount()} returns 0.</li>
 *     <li>{@link #canUserAccessApplication(com.atlassian.sal.api.user.UserKey)} returns {@code false}.</li>
 *     <li>{@link #getAccessError(com.atlassian.sal.api.user.UserKey)}} returns {@link AccessError#UNLICENSED}.</li>
 *     <li>{@link #getApplicationKey} return the constructor argument.</li>
 *     <li>{@link #getManagementPage} return the constructor argument.</li>
 * </ul>
 *
 * @since v1.0
 */
public final class NotLicensedAccess implements ApplicationAccess
{
    private final ApplicationKey key;
    private final URI managementPage;

    /**
     * Create a new instance with the passed key and management URI.
     *
     * @param key the key to return from {@link #getApplicationKey()}.
     * @param managementPage the URI to return from {@link #getManagementPage()}.
     */
    public NotLicensedAccess(final ApplicationKey key, final URI managementPage)
    {
        this.managementPage = Arguments.checkArgumentNotNull(managementPage, "managementPage");
        this.key = Arguments.checkArgumentNotNull(key, "key");
    }

    @Override
    public ApplicationKey getApplicationKey()
    {
        return key;
    }

    @Override
    public Option<Integer> getMaximumUserCount()
    {
        return Option.some(0);
    }

    @Override
    public int getActiveUserCount()
    {
        return 0;
    }

    @Override
    public boolean canUserAccessApplication(@Nullable final UserKey userKey)
    {
        return false;
    }

    @Override
    public Option<AccessError> getAccessError(@Nullable final UserKey userKey)
    {
        return Option.some(AccessError.UNLICENSED);
    }

    @Override
    public URI getManagementPage()
    {
        return managementPage;
    }
}
