package com.atlassian.application.host.plugin;

import com.atlassian.fugue.Option;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * Some utility methods related to parsing and processing {@code DateTime} instances.
 *
 * @since v1.0
 */
final class PluginDateTimeUtils
{
    private static final DateTimeFormatter BUILD_DATE_FORMAT
        = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZ").withOffsetParsed();

    /**
     * Cannot create instances.
     */
    private PluginDateTimeUtils()
    {
    }

    /**
     * Parse the passed string a plugin date and time. {@link com.atlassian.fugue.Option#none()} is returned
     * if the passed string is not a valid plugin date and time.
     *
     * @param date the string to parse.
     *
     * @return the parsed {@link org.joda.time.DateTime} or {@link com.atlassian.fugue.Option#none()} if the string
     * does not parse.
     */
    static Option<DateTime> fromPluginString(final String date)
    {
        try
        {
            return Option.some(BUILD_DATE_FORMAT.parseDateTime(date));
        }
        catch (final IllegalArgumentException e)
        {
            return Option.none();
        }
    }

    /**
     * Turn the passed {@link DateTime} into a String.
     *
     * @return the passed {@code DateTime} as a string.
     */
    static String toPluginString(final DateTime date)
    {
        return BUILD_DATE_FORMAT.print(date);
    }
}
