package com.atlassian.application.host.plugin;

import com.atlassian.application.host.events.ApplicationDefinedEvent;
import com.atlassian.application.host.events.ApplicationUndefinedEvent;
import com.atlassian.application.host.util.Arguments;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.fugue.Option;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.event.events.PluginModuleDisabledEvent;
import com.atlassian.plugin.event.events.PluginModuleEnabledEvent;

/**
 * Listens events from the host and transforms them into {@code Application} specific events.
 *
 * <p>
 *
 * @since v1.0
 */
public final class ApplicationEventListener
{
    private final EventPublisher publisher;

    /**
     * Create the event listener.
     *
     * @param publisher the bus to publish any events on.
     */
    public ApplicationEventListener(final EventPublisher publisher)
    {
        this.publisher = Arguments.checkArgumentNotNull(publisher, "publisher");
    }

    /**
     * Listens for when a plugin module is disabled and fires an
     * {@link com.atlassian.application.host.events.ApplicationUndefinedEvent} if it is relevant.
     *
     * @param event the event to check.
     */
    @EventListener
    public void pluginModuleDisabled(final PluginModuleDisabledEvent event)
    {
        for (ApplicationMetaDataModuleDescriptor licenseRoleModuleDescriptor : asRole(event.getModule()))
        {
            publisher.publish(new ApplicationUndefinedEvent(licenseRoleModuleDescriptor.getApplicationKey()));
        }
    }

    /**
     * Listens for when a plugin module is enabled and fires an
     * {@link com.atlassian.application.host.events.ApplicationDefinedEvent} if it is relevant.
     *
     * @param event the event to check.
     */
    @EventListener
    public void pluginModuleEnabled(final PluginModuleEnabledEvent event)
    {
        for (ApplicationMetaDataModuleDescriptor licenseRoleModuleDescriptor : asRole(event.getModule()))
        {
            publisher.publish(new ApplicationDefinedEvent(licenseRoleModuleDescriptor.getApplicationKey()));
        }
    }

    /**
     * Register the listener with the event bus.
     */
    public void register()
    {
        publisher.register(this);
    }

    private Option<ApplicationMetaDataModuleDescriptor> asRole(final ModuleDescriptor module)
    {
        if (module instanceof ApplicationMetaDataModuleDescriptor)
        {
            return Option.some((ApplicationMetaDataModuleDescriptor) module);
        }
        else
        {
            return Option.none(ApplicationMetaDataModuleDescriptor.class);
        }
    }
}
