package com.atlassian.application.api;

import com.atlassian.fugue.Option;

/**
 * Interface into the {@link Application}s (e.g. Service Desk) installed on the underlying platform (e.g. JIRA).
 *
 * Unless otherwise noted none of the parameters or return values can be {@code null}.
 *
 * @since v1.0
 */
public interface ApplicationManager
{
    /**
     * Return the {@link com.atlassian.application.api.PlatformApplication} that represents the Platform.
     *
     * @return the {@code PlatformApplication} that represents the Platform.
     */
    PlatformApplication getPlatform();

    /**
     * Return all the {@link Application}s installed on the platform. This will include the
     * {@link com.atlassian.application.api.PlatformApplication}. An application may or may not be licensed.
     *
     * @return all the {@code Application}s installed on the platform.
     */
    Iterable<Application> getApplications();

    /**
     * Return the installed {@link Application} with the passed key. {@link com.atlassian.fugue.Option#none()}
     * is returned to indicate that the {@code Application} is not installed.
     *
     * @param key the key to search for.
     * @return the {@code Application} with the passed key or {@link com.atlassian.fugue.Option#none()} if no such
     * {@code Application} exists.
     */
    Option<Application> getApplication(ApplicationKey key);

    /**
     * Return the installed {@link Application} associated with the passed
     * {@link ApplicationKey} if it is of the correct type. {@link com.atlassian.fugue.Option#none()}
     * is returned to indicate no match.
     *
     * @param key the key to search for.
     * @param type the type of {@code Application} to return.
     * @param <A> the type of {@code Application} to return.
     * @return the {@code Application} with the passed key and type or {@link com.atlassian.fugue.Option#none()} if
     * no such {@code Application} exists.
     */
    <A extends Application> Option<A> getApplication(ApplicationKey key, Class<A> type);

    /**
     * Get the {@link ApplicationAccess} associated with the passed application,
     * or {@link com.atlassian.fugue.Option#none()} if there is no {@code Application} installed with the passed key.
     *
     * @param key the key to search for.
     * @return the {@code ApplicationAccess} associated with the passed key.
     */
    Option<ApplicationAccess> getAccess(ApplicationKey key);
}
