package com.atlassian.application.api;

import java.io.Serializable;
import java.util.Locale;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Represents the identifier to an Application. An {@code ApplicationKey} must match the
 * the regular expression {@code [a-zA-Z.]+}. It is case-preserving but not case-sensitive. For example,
 * {@code ApplicationKey.valueOf("ABC").equals(ApplicationKey.valueOf("aBc"))} evaluates to {@code true}.
 *
 * @since 1.0
 */
public final class ApplicationKey implements Serializable
{
    private static final long serialVersionUID = 1L;

    /**
     * This is the regex that all keys must match. It is restricted to this pattern to ensure that the key can work
     * on a URL.
     */
    private static final Pattern ALLOWED_KEYS = Pattern.compile("[a-zA-Z.-]+");

    private final String key;
    private transient String comparableKey;

    private ApplicationKey(final String key)
    {
        if (!isValid(key))
        {
            throw new IllegalArgumentException("'key' must match the regular expression '" + ALLOWED_KEYS + "'.");
        }

        this.key = key;
    }

    /**
     * Return the {@code String} key used to create the ApplicationKey.
     *
     * @return the {@code String} key used to create the ApplicationKey.
     */
    @Nonnull
    public String value()
    {
        return this.key;
    }

    @Override
    public boolean equals(final Object x)
    {
        if (!(x instanceof ApplicationKey))
        {
            return false;
        }

        final ApplicationKey that = (ApplicationKey) x;
        return this.getComparableKey().equals(that.getComparableKey());
    }

    @Override
    public int hashCode()
    {
        return this.getComparableKey().hashCode();
    }

    @Override
    public String toString()
    {
        return value();
    }

    private String getComparableKey()
    {
        if (comparableKey == null)
        {
            comparableKey = key.toLowerCase(Locale.ENGLISH);
        }
        return comparableKey;
    }

    /**
     * Create a new {@link com.atlassian.application.api.ApplicationKey}. The passed key must match the
     * regex {@code [a-zA-Z.]+}.
     *
     * @param key the string for the {@code ApplicationKey}.
     * @return the new {@code ApplicationKey}
     * @throws java.lang.IllegalArgumentException if key is invalid.
     */
    public static ApplicationKey valueOf(final String key)
    {
        return new ApplicationKey(key);
    }

    /**
     * Indicate if the passed key is valid.
     *
     * @param key the key to check.
     * @return {@code true} if the passed key is valid.
     */
    public static boolean isValid(@Nullable final String key)
    {
        return key != null && ALLOWED_KEYS.matcher(key).matches();
    }
}