package com.atlassian.analytics.client.serialize;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.analytics.api.events.MauEvent;
import com.atlassian.analytics.client.api.browser.BrowserEvent;
import com.atlassian.analytics.client.extractor.PropertyExtractor;
import com.atlassian.analytics.client.hash.AnalyticsEmailHasher;
import com.atlassian.analytics.client.properties.AnalyticsPropertyService;
import com.atlassian.analytics.client.sen.SenProvider;
import com.atlassian.analytics.event.EventUtils;
import com.atlassian.analytics.event.RawEvent;
import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.sal.api.UrlMode;
import com.atlassian.util.concurrent.LazyReference;
import com.google.common.base.Optional;
import com.google.common.base.Supplier;
import com.google.common.collect.Maps;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;

import javax.annotation.Nullable;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import static com.google.common.base.Optional.fromNullable;

public class EventSerializer
{
    private static final String MAU_EMAIL_KEY = "email";
    private static final String MAU_EMAIL_HASH_KEY = "emailHash";
    protected static final String MAU_SUPPRESSED_USERNAME_VALUE = "suppressed";

    private final PropertyExtractor propertyExtractor;
    private final AnalyticsPropertyService applicationProperties;
    private final SenProvider senProvider;

    private final AtomicReference<String> server = new AtomicReference<String>();
    private final String product;
    private final String version;
    private final Cache<Optional<String>, Optional<String>> emailHashCache;
    private final LazyReference<Boolean> isMauEventAvailable = new IsMauEventAvailable();

    public EventSerializer(final AnalyticsPropertyService applicationProperties,
                           final PropertyExtractor propertyExtractor,
                           final SenProvider senProvider,
                           final CacheManager cacheManager,
                           final AnalyticsEmailHasher analyticsEmailHasher)
    {
        this.applicationProperties = applicationProperties;
        this.product = applicationProperties.getDisplayName().toLowerCase();
        this.version = applicationProperties.getVersion();
        this.propertyExtractor = propertyExtractor;
        this.senProvider = senProvider;
        this.emailHashCache = cacheManager.getCache(EventSerializer.class.getName() + ".emailHash",
                    new CacheLoader<Optional<String>, Optional<String>>()
                {
                    @Override
                    public Optional<String> load(Optional<String> email)
                    {
                        return email.isPresent()
                                ? fromNullable(analyticsEmailHasher.hash(email.get()))
                                : Optional.<String>absent();
                    }
                },
                new CacheSettingsBuilder()
                        .local()
                        .expireAfterAccess(2, TimeUnit.HOURS).build());
    }

    public Supplier<RawEvent> toAnalyticsEvent(final Object event, final @Nullable String sessionId, final RequestInfo
        requestInfo)
    {
        return new Supplier<RawEvent>()
        {
            private final RawEvent.Builder builder;
            private final Map<String, Object> properties;

            {
                final String originalName = propertyExtractor.extractName(event);
                final Map<String, Object> originalProperties = extractEventProperties(event);

                final String name = EventUtils.getBrowserEventName(originalName, originalProperties);
                properties = EventUtils.getBrowserEventProperties(originalName, originalProperties);

                String subProduct = propertyExtractor.extractSubProduct(event, product);

                final String server = getServer();
                final long currentTime = getCurrentTime();
                final long clientTime = getClientTime(event).or(currentTime);
                final String user = extractUser();
                final String applicationAccess = propertyExtractor.getApplicationAccess();
                final String requestCorrelationId = extractRequestCorrelationId(requestInfo);

                // We use the hash of the session ID instead of the raw session ID string for privacy reasons (i.e. so users
                // sessions can't be hijacked)
                final String session = sessionId != null ? Integer.toString(sessionId.hashCode()) : null;

                builder = new RawEvent.Builder()
                        .name(name)
                        .server(server)
                        .product(product)
                        .subproduct(subProduct)
                        .version(version)
                        .user(user)
                        .session(session)
                        .clientTime(clientTime)
                        .receivedTime(currentTime)
                        .sourceIP(requestInfo.getSourceIp())
                        .atlPath(requestInfo.getAtlPath())
                        .appAccess(applicationAccess)
                        .requestCorrelationId(requestCorrelationId);
            }

            private String extractUser()
            {
                if (isMauEvent(event))
                {
                    return MAU_SUPPRESSED_USERNAME_VALUE;
                }
                return propertyExtractor.extractUser(event, properties);
            }

            private String extractRequestCorrelationId(RequestInfo requestInfo)
            {
                if (isMauEvent(event))
                {
                    return "";
                }
                return propertyExtractor.extractRequestCorrelationId(requestInfo);
            }

            @Override
            public RawEvent get()
            {
                final Map<String, Object> eventProperties = hashEmailPropertyForMauEvent(event, properties);
                //We need to retrieve the SEN on a separate thread, doing it on the same event thread 
                // can cause a deadlock when bitbucket server runs as a mirror. Don't move this outside the `get` block
                final String sen = senProvider.getSen();
                return builder.properties(eventProperties).sen(sen).build();
            }
        };
    }

    private Map<String, Object> hashEmailPropertyForMauEvent(Object event, Map<String, Object> properties)
    {
        if (isMauEvent(event))
        {
            final Map<String, Object> eventProperties = Maps.newHashMap(properties);
            final Optional<String> emailHash = emailHashCache.get(fromNullable((String)eventProperties.get(MAU_EMAIL_KEY)));
            if (emailHash.isPresent())
            {
                eventProperties.put(MAU_EMAIL_HASH_KEY, emailHash.get());
            }
            eventProperties.remove(MAU_EMAIL_KEY);
            return eventProperties;
        }
        return properties;
    }

    private boolean isMauEvent(Object event)
    {
        return isMauEventAvailable.get() && event instanceof MauEvent;
    }

    private long getCurrentTime()
    {
        return System.currentTimeMillis();
    }

    private Optional<Long> getClientTime(Object event)
    {
        if (event instanceof BrowserEvent)
        {
            return Optional.of(((BrowserEvent) event).getClientTime());
        }
        else
        {
            return Optional.absent();
        }
    }

    public Map<String, Object> extractEventProperties(Object event)
    {
        Map<String, Object> result = Maps.newHashMap();
        BeanWrapper beanWrapper = new BeanWrapperImpl(event);
        for (PropertyDescriptor property : beanWrapper.getPropertyDescriptors())
        {
            String name = property.getName();
            if (!propertyExtractor.isExcluded(name) && !isEventName(property))
            {
                Object value = beanWrapper.getPropertyValue(name);
                result.putAll(propertyExtractor.extractProperty(name, value));
            }
        }
        result.putAll(propertyExtractor.enrichProperties(event));
        return result;
    }

    private boolean isEventName(final PropertyDescriptor property)
    {
        final Method readMethod = property.getReadMethod();
        if (readMethod != null)
        {
            for (Annotation annotation : readMethod.getDeclaredAnnotations())
            {
                if (annotation.annotationType().equals(EventName.class))
                {
                    return true;
                }
            }
        }
        return false;
    }

    // caches the first non-null base URL set in the instance
    private String getServer()
    {
        String result = server.get();
        if (result != null) return result;

        String baseUrl = applicationProperties.getBaseUrl(UrlMode.CANONICAL);
        if (baseUrl == null) return "-";
        try
        {
            result = new URL(baseUrl).getHost();
            server.set(result);
            return result;
        }
        catch (MalformedURLException e)
        {
            return "-";
        }
    }
}
