package com.atlassian.analytics.client.sender;

import com.atlassian.analytics.client.configuration.AnalyticsConfig;
import com.atlassian.analytics.event.ProcessedEvent;
import com.atlassian.analytics.event.transport.UDPSender;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.io.IOException;
import java.net.InetAddress;

class EventSender implements DisposableBean
{
    private static final String ANALYTICS_SERVER_DEFAULT = System.getProperty("analytics.server", "analytics");
    private static final int ANALYTICS_PORT_DEFAULT = 19876;

    private static final Logger log = LoggerFactory.getLogger(EventSender.class);

    private UDPSender udpSender;
    private InetAddress address;
    private int port;
    private final String hostname;

    public static EventSender newInstance(AnalyticsConfig analyticsConfig)
    {
        String destination = analyticsConfig.getDestinationOrDefault(ANALYTICS_SERVER_DEFAULT);
        return new EventSender(getHostname(destination), getPort(destination, ANALYTICS_PORT_DEFAULT));
    }

    private static String getHostname(String serverName)
    {
        int colonIndex = serverName.indexOf(":");
        return colonIndex > 0 ? serverName.substring(0, colonIndex) : serverName;
    }

    private static int getPort(String serverName, int defaultPort)
    {
        int colonIndex = serverName.indexOf(":");
        return colonIndex > 0 ? Integer.parseInt(serverName.substring(colonIndex + 1)) : defaultPort;
    }

    public EventSender(String hostname, int port)
    {
        this.hostname = hostname;
        this.port = port;
        try
        {
            udpSender = new UDPSender();
            address = InetAddress.getByName(hostname);
        }
        catch (IOException e)
        {
            log.debug("Unable to open connection to analytics server. Event transmission disabled. " + e.getMessage());
            address = null;
            udpSender = null;
        }
    }

    public void destroy()
    {
        if (udpSender != null)
        {
            try
            {
                udpSender.shutdown();
            }
            catch (Exception e)
            {
                log.error("Failed to shut down UDP sender on port " + port + ": " + e.getMessage(), e);
            }
            udpSender = null;
        }
        address = null;
    }

    public void send(ProcessedEvent event)
    {
        if (udpSender == null || address == null)
            return;
        long start = System.nanoTime();
        try
        {
            udpSender.send(event.toEventMessage(), address, port);
        }
        catch (IOException e)
        {
            log.error("Failed to send event [" + event + "]: " + e.getMessage());
        }
        finally
        {
            log.debug("Sending event message {} took {} \u00b5s", event.getName(), (System.nanoTime() - start) / 1000);
        }
    }

    @Override
    public String toString()
    {
        return String.format("EventSender[%s:%d%s]", hostname, port, udpSender == null ? " (disabled)" : "");
    }
}
