package com.atlassian.analytics.client.logger;

import com.atlassian.analytics.client.AnalyticsMd5Hasher;
import com.atlassian.analytics.client.ServerIdProvider;
import com.atlassian.analytics.client.cluster.ClusterInformationProvider;
import com.atlassian.analytics.client.properties.AnalyticsPropertyService;
import com.atlassian.analytics.client.sen.SenProvider;
import com.atlassian.analytics.event.ProcessedEvent;
import com.atlassian.analytics.event.logging.LogEventFormatter;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.springframework.beans.factory.DisposableBean;
import uk.org.simonsite.log4j.appender.TimeAndSizeRollingAppender;

import java.io.File;
import java.io.IOException;

public class Log4jAnalyticsLogger implements AnalyticsLogger, DisposableBean
{
    private static final Logger LOG = Logger.getLogger(Log4jAnalyticsLogger.class);

    public static final String ANALYTICS_LOGS_DIR = "analytics-logs";
    public static final String ATLASSIAN_ANALYTICS_LOG_FILENAME = ".atlassian-analytics.log";
    public static final String ANALYTICS_LOG_ROLLING_DATE_PATTERN = "'.'yyyy-MM-dd";

    private static final String ANALYTICS_LOGGER_NAME = "com.atlassian.analytics.client.btflogger";
    private static final String GZIP_COMPRESSION_ALGORITHM = "GZ";
    private static final String UTC_TIMEZONE_ID = "UTC";

    private final Logger analyticsLog;

    private final LogEventFormatter logEventFormatter;
    private final AnalyticsPropertyService applicationProperties;

    private TimeAndSizeRollingAppender analyticsRollingAppender;
    private String logPath;

    
    public Log4jAnalyticsLogger(final LogEventFormatter logEventFormatter, final AnalyticsPropertyService applicationProperties,
                                final SenProvider senProvider, final ServerIdProvider serverIdProvider,
                                final ClusterInformationProvider clusterInformationProvider)
    {
        this.logEventFormatter = logEventFormatter;
        this.applicationProperties = applicationProperties;
        this.analyticsLog = Logger.getLogger(ANALYTICS_LOGGER_NAME);

        // Generate a unique server ID to be the prefix for the log filename
        String uniqueServerId = generateUniqueServerId(senProvider.getSen(),
                                                       serverIdProvider.getServerId(),
                                                       clusterInformationProvider.getCurrentNodeId());

        this.logPath = generateLogPath(uniqueServerId);

        initAnalyticsLogger();
        initAnalyticsAppender();
    }

    private String generateLogPath(final String uniqueServerId)
    {
        final File logDirPath = getAbsoluteLogDirPath(applicationProperties);

        if (!logDirPath.exists())
        {
            if (!logDirPath.mkdir())
            {
                LOG.error("Couldn't create a directory for analytics logging.");
            }
        }

        return new File(logDirPath.getAbsolutePath(), (uniqueServerId + ATLASSIAN_ANALYTICS_LOG_FILENAME)).getAbsolutePath();
    }

    private String generateUniqueServerId(final String sen, final String serverId, final String currentNodeId)
    {
        return AnalyticsMd5Hasher.md5Hex(StringUtils.defaultString(serverId) +
                                         StringUtils.defaultString(sen)) +
               (StringUtils.isNotBlank(currentNodeId) ? ("." + currentNodeId) : "");
    }

    private void initAnalyticsAppender()
    {
        analyticsRollingAppender = new TimeAndSizeRollingAppender();
        analyticsRollingAppender.setName(ANALYTICS_LOGGER_NAME);
        analyticsRollingAppender.setFile(logPath);
        analyticsRollingAppender.setDatePattern(ANALYTICS_LOG_ROLLING_DATE_PATTERN);
        analyticsRollingAppender.setCompressionAlgorithm(GZIP_COMPRESSION_ALGORITHM);
        analyticsRollingAppender.setTimeZoneId(UTC_TIMEZONE_ID);
        analyticsRollingAppender.setAppend(true);
        analyticsRollingAppender.setLayout(new PatternLayout("%m%n"));
        analyticsRollingAppender.setThreshold(Level.DEBUG);
        analyticsRollingAppender.setRollOnStartup(true);
        analyticsRollingAppender.activateOptions();

        analyticsLog.addAppender(analyticsRollingAppender);
    }

    private void initAnalyticsLogger()
    {
        analyticsLog.setAdditivity(false);
        analyticsLog.setLevel(Level.INFO);
    }

    @Override
    public void logEvent(ProcessedEvent event)
    {
        try
        {
            analyticsLog.info(logEventFormatter.formatEvent(event));
        }
        catch (IOException e)
        {
            LOG.debug("Couldn't log event information to file, failed to serialize the event properties.");
        }
    }

    @Override
    public void logCleanupDeletion(String deletionMessage)
    {
        analyticsLog.debug("Deleted " + deletionMessage);
    }

    @Override
    public void destroy() throws Exception
    {
        removeAppenders();
    }

    public static File getAbsoluteLogDirPath(final AnalyticsPropertyService applicationProperties)
    {
        final String homePath = applicationProperties.getHomeDirectory().getAbsolutePath();
        return new File(homePath, ANALYTICS_LOGS_DIR);
    }

    @Override
    public void reset()
    {
        removeAppenders();
        initAnalyticsAppender();
    }

    private void removeAppenders()
    {
        // Make sure that we remove all references to the appender so it can be garbage collected
        analyticsRollingAppender.close();
        analyticsLog.removeAllAppenders();
        analyticsRollingAppender = null;
    }
}
