package com.atlassian.analytics.client.hash;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.commons.lang.StringUtils.remove;
import static org.apache.commons.lang.StringUtils.substringBefore;

/**
 * Cleans an e-mail address for hashing.  Cleaning involves the following steps:
 * <ul>
 * <li>verifying the e-mail is not blank</li>
 * <li>verifying the e-mail contains @</li>
 * <li>replacing any . characters in the first part before @</li>
 * <li>removing any characters after + in the first part before @</li>
 * <li>lowercasing the entire e-mail address</li>
 * </ul>
 *
 * @since v4.3
 */
public class EmailCleaner {
    private static final Logger LOG = LoggerFactory.getLogger(EmailCleaner.class);

    public static String cleanEmail(String email) {
        if (StringUtils.isBlank(email)) {
            return null;
        }

        if (!email.contains("@")) {
            LOG.warn("email is invalid '{0}'. returning empty string", email);
            return null;
        }

        // 'clean' the email
        final String[] emailParts = email.toLowerCase().split("@");
        if (emailParts.length != 2) {
            LOG.warn("email is invalid '{0}'. returning empty string", email);
            return null;
        }

        final String firstPartWithoutPlus = substringBefore(emailParts[0], "+");
        final String firstPartWithoutDot = remove(firstPartWithoutPlus, ".");
        return firstPartWithoutDot + "@" + emailParts[1];
    }
}
