package com.atlassian.analytics.client.eventfilter.whitelist;

import com.google.common.collect.ImmutableMap;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.annotate.JsonProperty;

import java.util.HashMap;
import java.util.Map;

/**
 * Standard analytics whitelist, describing a map of event names to filtered attributes. Events found in the whitelist
 * will be allowed to proceed through the analytics pipeline. Attributes found listed for events will also be allowed
 * through, processed according to the list they are found in.
 */
public class Whitelist
{
    private String whitelistId;
    private final boolean isGlobalWhitelist;

    /*
     * This list is never mutated, but always replaced by a completely new one. To ensure there are no concurrent
     * modifications, it is an ImmutableList. To ensure that other threads see the updated reference, it is volatile.
     */
    private volatile ImmutableMap<String, FilteredEventAttributes> filteredEventAttributesMap = ImmutableMap.of();

    public static Whitelist createEmptyWhitelist()
    {
        return new Whitelist("", new HashMap<String, FilteredEventAttributes>(), false);
    }

    public Whitelist(final String whitelistId, final Map<String, FilteredEventAttributes> filteredEventAttributesMap, final boolean isGlobalWhitelist)
    {
        this.isGlobalWhitelist = isGlobalWhitelist;
        initialiseFrom(whitelistId, filteredEventAttributesMap);
    }

    public void initialiseFrom(final Whitelist globalWhitelist)
    {
        initialiseFrom(globalWhitelist.whitelistId, globalWhitelist.filteredEventAttributesMap);
    }

    private void initialiseFrom(final String whitelistId, final Map<String, FilteredEventAttributes> filteredEventAttributesMap)
    {
        this.whitelistId = whitelistId;
        initialiseFrom(filteredEventAttributesMap);
    }

    public void initialiseFrom(final Map<String, FilteredEventAttributes> filteredEventAttributesMap)
    {
        if (filteredEventAttributesMap != null)
        {
            this.filteredEventAttributesMap = ImmutableMap.copyOf(filteredEventAttributesMap);
        }
    }

    public boolean isEventWhitelisted(final String eventName)
    {
        return filteredEventAttributesMap.containsKey(eventName);
    }

    public boolean shouldAttributeBeHashed(final String eventName, final String attributeName)
    {
        FilteredEventAttributes filteredEventAttributes = filteredEventAttributesMap.get(eventName);
        return filteredEventAttributes != null && filteredEventAttributes.hasHashedAttribute(attributeName);
    }

    public boolean shouldAttributeBeDictionaryFiltered(final String eventName, final String attributeName)
    {
        FilteredEventAttributes filteredEventAttributes = filteredEventAttributesMap.get(eventName);
        return filteredEventAttributes != null && filteredEventAttributes.hasDictionaryFilteredAttribute(attributeName);
    }

    public boolean shouldAttributeBeWhitelisted(final String eventName, final String attributeName)
    {
        FilteredEventAttributes filteredEventAttributes = filteredEventAttributesMap.get(eventName);
        return filteredEventAttributes != null && filteredEventAttributes.hasWhitelistedAttribute(attributeName);
    }

    FilteredEventAttributes getFilteredEventAttributes(final String eventName)
    {
        return filteredEventAttributesMap.get(eventName);
    }

    public Map<String, FilteredEventAttributes> getFilteredEventAttributesMap()
    {
        return filteredEventAttributesMap;
    }

    public boolean isGlobalWhitelist()
    {
        return isGlobalWhitelist;
    }

    public String getWhitelistId()
    {
        return whitelistId;
    }

    public WhitelistBean toWhitelistBean()
    {
        return new WhitelistBean(this);
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class WhitelistBean
    {
        @JsonProperty
        private final String whitelistId;

        @JsonProperty
        private final Map<String, FilteredEventAttributes> filteredEvents;

        public WhitelistBean(Whitelist whitelist)
        {
            this.whitelistId = whitelist.whitelistId;
            this.filteredEvents = whitelist.filteredEventAttributesMap;
        }
    }
}
