package com.atlassian.analytics.client.eventfilter.whitelist;

import com.google.common.collect.ImmutableList;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Represents a whitelist aggregated from all externally defined whitelists (i.e. plugin-descriptor whitelists).
 */
public class AggregatedWhitelist
{
    /*
     * This list is never mutated, but always replaced by a completely new one. To ensure there are no concurrent
     * modifications, it is an ImmutableList. To ensure that other threads see the updated reference, it is volatile.
     */
    private volatile ImmutableList<Whitelist> externalWhitelists = ImmutableList.of();

    public static AggregatedWhitelist createEmptyAggregate()
    {
        return new AggregatedWhitelist();
    }

    public void initialiseFrom(final List<Whitelist> externalWhitelists)
    {
        this.externalWhitelists = ImmutableList.copyOf(externalWhitelists);
    }

    public boolean isEventWhitelisted(final String eventName)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            if (whitelist.getFilteredEventAttributes(eventName) != null)
            {
                return true;
            }
        }
        return false;
    }

    public boolean isAttributeValueInAllowedList(String eventName, String attributeName, String attributeValue)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            FilteredEventAttributes filteredEventAttributes = whitelist.getFilteredEventAttributes(eventName);
            if (filteredEventAttributes != null && filteredEventAttributes.hasAllowedAttributeValue(attributeName, attributeValue))
            {
                return true;
            }
        }
        return false;
    }

    public boolean shouldAttributeBeHashed(final String eventName, final String attributeName)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            FilteredEventAttributes filteredEventAttributes = whitelist.getFilteredEventAttributes(eventName);
            if (filteredEventAttributes != null && filteredEventAttributes.hasHashedAttribute(attributeName))
            {
                return true;
            }
        }
        return false;
    }

    public boolean shouldAttributeBeDictionaryFiltered(final String eventName, final String attributeName)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            FilteredEventAttributes filteredEventAttributes = whitelist.getFilteredEventAttributes(eventName);
            if (filteredEventAttributes != null && filteredEventAttributes.hasDictionaryFilteredAttribute(attributeName))
            {
                return true;
            }
        }
        return false;
    }

    public boolean shouldAttributeBeWhitelisted(final String eventName, final String attributeName)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            FilteredEventAttributes filteredEventAttributes = whitelist.getFilteredEventAttributes(eventName);
            if (filteredEventAttributes != null && filteredEventAttributes.hasWhitelistedAttribute(attributeName))
            {
                return true;
            }
        }
        return false;
    }

    public List<Whitelist> getWhitelists()
    {
        return externalWhitelists;
    }

    public Collection<? extends Whitelist.WhitelistBean> toWhitelistBeans()
    {
        final List<Whitelist.WhitelistBean> whitelistBeans = new ArrayList<Whitelist.WhitelistBean>(externalWhitelists.size());
        for (Whitelist externalWhitelist : externalWhitelists)
        {
            whitelistBeans.add(new Whitelist.WhitelistBean(externalWhitelist));
        }
        return whitelistBeans;
    }
}
