package com.atlassian.analytics.api.events;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.event.api.AsynchronousPreferred;

/**
 * Represents an eyeball event used for tracking monthly active users in Atlassian applications.
 * The email address provided will be securely hashed before this event will be sent to Atlassian.
 * The application field is optional and should only be filled in by applications that can have multiple
 * applications licensed within a single instance. For example any given JIRA instance could have jira-core,
 * jira-software or jira-servicedesk licensed and installed.
 *
 * @since v4.3
 */
@EventName(MauEvent.EVENT_NAME)
@AsynchronousPreferred
public final class MauEvent {
    public static final String EVENT_NAME = "UserActivity";

    private final String email;
    private final String application;

    private MauEvent(final String email, final String application) {
        this.email = email;
        this.application = application;
    }

    public String getEmail() {
        return email;
    }

    public String getApplication() {
        return application;
    }

    public static MauEvent withEmail(String email) {
        return new MauEvent.Builder().build(email);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        MauEvent mauEvent = (MauEvent) o;

        if (email != null ? !email.equals(mauEvent.email) : mauEvent.email != null) return false;
        return !(application != null ? !application.equals(mauEvent.application) : mauEvent.application != null);

    }

    @Override
    public int hashCode() {
        int result = email != null ? email.hashCode() : 0;
        result = 31 * result + (application != null ? application.hashCode() : 0);
        return result;
    }

    public static class Builder {
        private String application;

        public Builder application(String application) {
            this.application = application;
            return this;
        }

        public MauEvent build(String email) {
            if (email == null) {
                throw new IllegalArgumentException("Email cannot be null.");
            }
            return new MauEvent(email, this.application);
        }
    }
}
