package com.artear.ui.base

import android.os.Bundle
import androidx.fragment.app.Fragment
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import com.artear.tools.error.NestError
import com.artear.ui.interfaces.UIStateFlow
import com.artear.ui.model.State
import com.artear.ui.viewmodel.StateViewModel


abstract class ArtearFragment : Fragment(), UIStateFlow {

    companion object {
        const val FRAGMENT_TITLE = "artear_fragment_title"
        const val FRAGMENT_TITLE_RES = "artear_fragment_title_resource"
        const val FRAGMENT_ID = "fragment_id"
    }

    lateinit var id: String

    var title: String? = null

    var titleResourceId: Int? = null
        set(value) {
            value?.let {
                title = getString(it)
                field = it
            }
        }

    lateinit var uiStateViewModel: StateViewModel

    private val observer = Observer<State> { onStateChange(it) }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val bundle = savedInstanceState?: arguments
        restoreProperties(bundle)
        uiStateViewModel = ViewModelProviders.of(this)[StateViewModel::class.java]
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        outState.putString(FRAGMENT_ID, id)
        outState.putString(FRAGMENT_TITLE, title)
        titleResourceId?.let {
            outState.putInt(FRAGMENT_TITLE_RES, it)
        }
    }

    override fun onViewStateRestored(savedInstanceState: Bundle?) {
        super.onViewStateRestored(savedInstanceState)
        restoreProperties(savedInstanceState)
    }

    private fun restoreProperties(bundle: Bundle?) {
        bundle?.apply {
            getString(FRAGMENT_ID)?.let { id = it }
            getString(FRAGMENT_TITLE)?.let { title = it }
            if (getInt(FRAGMENT_TITLE_RES) != 0) {
                titleResourceId = getInt(FRAGMENT_TITLE_RES)
            }
        }
    }

    override fun onActivityCreated(savedInstanceState: Bundle?) {
        super.onActivityCreated(savedInstanceState)
        uiStateViewModel.state.observe(this, observer)
    }

    protected fun onStateChange(state: State?) {
        state?.let {
            when (it) {
                is State.Loading -> onLoading()
                is State.Success -> onSuccess()
                is State.Error -> onError(it.error)
            }
        }
    }

    override fun onLoading() {
    }

    override fun onSuccess() {
    }

    override fun onError(error: NestError) {
    }

    override fun onDestroyView() {
        super.onDestroyView()
        uiStateViewModel.state.removeObserver(observer)
    }

}

