@file:Suppress("unused")

package com.appspiriment.permissionutils

import android.content.DialogInterface
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import kotlinx.coroutines.*


/*********************************************************
 * Class   :  PermissionManager
 * Author  :  Arun Nair
 * Created :  14/10/20
 *******************************************************
 * Purpose :
 *******************************************************
 * Rework Details:
 * 1) {Author} :  {Date} : {Details}
 *********************************************************/

object PermissionManager {
    private const val TAG = "PermissionManager"
    private val parentJob = Job()
    private val coroutineScope = CoroutineScope(parentJob + Dispatchers.Default)

    /*********************************
     * Create a static coroutineScope
     * and return [CoroutineScope]
     ********************************/
    fun getCoroutineScope(): CoroutineScope {
        return coroutineScope
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissions : Permissions VarArg
     ******************************************/
    suspend fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return requestPermissionsFromUser(
            supportFragmentManager,
            requestId,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissions : Permissions VarArg
     ******************************************/
    suspend fun Fragment.requestPermissionsAsync(
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return requestPermissionsFromUser(
            childFragmentManager,
            requestId,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionGrantedListener
     * @permissionDeniedListener
     * @permissionDeniedPermanently
     * @permissionRationalListener
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit = {},
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit = { _, _ -> },
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit = { _, _ -> },
        permissionRationalListener: (requestId: Int) -> Unit = {},
        requestPermissionOnRationale: Boolean,
        vararg permissions: String
    ) {
        requestPermissionsFromUser(
            supportFragmentManager,
            requestId,
            permissionGrantedListener,
            permissionDeniedListener,
            permissionDeniedPermanently,
            permissionRationalListener,
            requestPermissionOnRationale,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionGrantedListener
     * @permissionDeniedListener
     * @permissionDeniedPermanently
     * @permissionRationalListener
     * @permissions
     ******************************************/
    fun Fragment.requestPermissionsAsync(
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit = {},
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit = { _, _ -> },
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit = { _, _ -> },
        permissionRationalListener: (requestId: Int) -> Unit = {},
        requestPermissionOnRationale: Boolean = true,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                childFragmentManager,
                requestId,
                permissionGrantedListener,
                permissionDeniedListener,
                permissionDeniedPermanently,
                permissionRationalListener,
                requestPermissionOnRationale,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListener
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionListener: PermissionListener,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                supportFragmentManager, requestId,
                permissionListener::onPermissionGranted,
                permissionListener::onPermissionDenied,
                permissionListener::onPermissionDeniedPermanently,
                permissionListener::onShowRational,
                false,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListener
     * @permissions
     ******************************************/
    fun Fragment.requestPermissionsAsync(
        requestId: Int,
        permissionListener: PermissionListener,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                childFragmentManager, requestId,
                permissionListener::onPermissionGranted,
                permissionListener::onPermissionDenied,
                permissionListener::onPermissionDeniedPermanently,
                permissionListener::onShowRational,
                false,
                *permissions
            )
        }
    }


    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListener
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionsList: List<String>,
        permissionListener: PermissionResultListener,
        rationaleTitle: String? = null,
        rationaleMessage: String? = null,
        rationaleTitleResId: Int? = null,
        rationaleMessageResId: Int? = null
    ) {
        val showRationaleLambda = { _: Int ->
            showRationale(
                this,
                rationaleTitle,
                rationaleTitleResId,
                rationaleMessage,
                rationaleMessageResId,
                requestId,
                permissionListener,
                *permissionsList.toTypedArray()
            )
        }
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                supportFragmentManager, requestId,
                permissionListener::onPermissionGranted,
                permissionListener::onPermissionDenied,
                permissionListener::onPermissionDeniedPermanently,
                showRationaleLambda,
                false,
                *permissionsList.toTypedArray()
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListener
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionsList: List<String>,
        permissionGrantedListener: (requestId: Int) -> Unit,
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        rationaleTitle: String? = null,
        rationaleMessage: String? = null,
        rationaleTitleResId: Int? = null,
        rationaleMessageResId: Int? = null
    ) {
        object : PermissionResultListener {
            override fun onPermissionGranted(requestCode: Int) {
                permissionGrantedListener(requestCode)
            }

            override fun onPermissionDenied(
                requestCode: Int,
                deniedPermissions: List<String>
            ) {
                permissionDeniedListener(requestCode, deniedPermissions)
            }

            override fun onPermissionDeniedPermanently(
                requestCode: Int,
                permanentlyDeniedPermissions: List<String>
            ) {
                permissionDeniedPermanently(requestCode, permanentlyDeniedPermissions)
            }
        }.let {
            requestPermissionsAsync(
                requestId,
                permissionsList,
                it,
                rationaleTitle,
                rationaleMessage,
                rationaleTitleResId,
                rationaleMessageResId
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *******************************************/
    private fun requestPermissionsFromUser(
        fragmentManager: FragmentManager,
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit,
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionRationalListener: (requestId: Int) -> Unit,
        requestPermissionOnRationale: Boolean = false,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {

            requestPermissionsFromUser(fragmentManager, requestId, *permissions).let {
                MainScope().launch {
                    when (it) {
                        is PermissionResult.PermissionGranted -> permissionGrantedListener(it.requestCode)
                        is PermissionResult.PermissionDenied -> permissionDeniedListener(
                            it.requestCode,
                            it.deniedPermissions
                        )
                        is PermissionResult.PermissionDeniedPermanently -> permissionDeniedPermanently(
                            it.requestCode,
                            it.permanentlyDeniedPermissions
                        )
                        is PermissionResult.ShowRational -> {
                            if (requestPermissionOnRationale) requestPermissionsFromUser(
                                fragmentManager, requestId,
                                permissionGrantedListener,
                                permissionDeniedListener,
                                permissionDeniedPermanently,
                                permissionRationalListener,
                                false,
                                *permissions
                            ) else permissionRationalListener(it.requestCode)
                        }
                    }
                }
            }
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *******************************************/
    private suspend fun requestPermissionsFromUser(
        fragmentManager: FragmentManager,
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return withContext(Dispatchers.Main) {
            val permissionFragment =
                fragmentManager.findFragmentByTag(TAG) ?: PermissionFragment().apply {
                    fragmentManager.beginTransaction().add(this, TAG).commitNow()
                }

            permissionFragment.let {
                (it as PermissionFragment).run {
                    deferredPermission = CompletableDeferred()
                    requestPermissionsFromUser(requestId, *permissions)
                    deferredPermission.await()
                }
            }
        }
    }

    /******************************************
     * Show Rationale for Permissions
     *******************************************/
    private fun showRationale(
        activity: AppCompatActivity,
        titleText: String?,
        titleResId: Int?,
        msgText: String?,
        msgResId: Int?,
        requestId: Int,
        permissionListener: PermissionResultListener,
        vararg permissions: String
    ) {

        val title = titleText ?: titleResId?.let { activity.resources.getString(it) }
        ?: throw  Exception("Please provide either rationaleTitleText or rationaleTitleResId!")

        val message = msgText ?: msgResId?.let { activity.resources.getString(it) }
        ?: throw  Exception("Please provide either rationaleMessage or rationaleResId!")

        val onClickListener = DialogInterface.OnClickListener { dialog, which ->
            dialog.dismiss()

            when (which) {
                DialogInterface.BUTTON_POSITIVE -> {
                    activity.requestPermissionsAsync(
                        requestId,
                        permissionListener::onPermissionGranted,
                        permissionListener::onPermissionDenied,
                        permissionListener::onPermissionDeniedPermanently,
                        {}, false,
                        *permissions
                    )
                }
                DialogInterface.BUTTON_NEGATIVE -> {
                    run {
                        permissionListener.onPermissionDeniedPermanently(
                            requestId,
                            emptyList()
                        )
                    }
                }
            }
        }

        val onCancelListener = DialogInterface.OnCancelListener { dialog ->
            dialog.dismiss()
            run { permissionListener.onPermissionDeniedPermanently(requestId, emptyList()) }
        }

        MainScope().launch {
            AlertDialog.Builder(activity).apply {
                setTitle(title)
                setMessage(message)
                setCancelable(true)
                setPositiveButton("Allow", onClickListener)
                setNegativeButton("Deny", onClickListener)
                setOnCancelListener(onCancelListener)
            }.create().show()
        }
    }
}
