package com.appspiriment.androidutils

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import com.androidnetworking.AndroidNetworking
import com.androidnetworking.common.Priority
import com.androidnetworking.error.ANError
import com.androidnetworking.interfaces.JSONArrayRequestListener
import com.androidnetworking.interfaces.JSONObjectRequestListener
import com.androidnetworking.interfaces.StringRequestListener
import okhttp3.OkHttpClient
import org.json.JSONArray
import org.json.JSONObject
import org.koin.core.KoinComponent
import java.util.concurrent.TimeUnit


/********************************************
 * Class for methods related actionTo Device
 *******************************************/
class NetworkUtils(val context: Context) : KoinComponent {
    companion object {
        val API_REQUEST_STRING = 101
        val API_REQUEST_JSON = 102
        val API_REQUEST_JSON_ARRAY = 103

        val RESPONSE_NONET = -101
        val RESPONSE_CUSTOM = -102

        /**
         * ******************************************
         * Method actionTo check whether the Internet is Connected
         * ******************************************
         */
        fun isNetworkConnected(context: Context): Boolean {
            val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
            if (Build.VERSION.SDK_INT < 23) {
                cm.activeNetworkInfo?.run {
                    return this.isConnected && (this.type == ConnectivityManager.TYPE_WIFI
                            || this.type == ConnectivityManager.TYPE_MOBILE)

                }

            } else {
                cm.getNetworkCapabilities(cm.activeNetwork)?.run {
                    return this.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) ||
                            this.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) ||
                            this.hasTransport(NetworkCapabilities.TRANSPORT_VPN)
                }
            }
            return false
        }
    }

    init {
        AndroidNetworking.initialize(context)
    }

    /**
     * ******************************************
     * Method actionTo start Volley Request
     * ******************************************
     */
    fun startApiRequest(
        url: String, stringSuccessListener: (String) -> Unit = {},
        jsonSuccessListener: (JSONObject) -> Unit = {},
        jsonArraySuccessListener: (JSONArray) -> Unit = {},
        errorListen: (errorCode: Int, errorDetail: String) -> Unit = { Int, String -> run {} },
        errorLogger: (error: ANError) -> Unit = {},
        requestType: Int = API_REQUEST_STRING
    ) {

        val errorHandler = { error: ANError ->
            error.run {
                errorListen(errorCode, errorBody ?: errorDetail ?: "Unexpected Error!")
                errorLogger(this)
            }
        }

        if (!isNetworkConnected(context)) {
            errorHandler(ANError().apply {
                errorBody = "No internet connection!"
                errorCode = RESPONSE_NONET
            })
            return
        }


        LogUtils.printLog(" URL   $url")
        val okHttpClient = OkHttpClient().newBuilder()
            .connectTimeout(3, TimeUnit.SECONDS)
            .readTimeout(5, TimeUnit.SECONDS)
            .writeTimeout(3, TimeUnit.SECONDS)
            .build()

        AndroidNetworking.get(url)
            .setPriority(Priority.MEDIUM)
            .setOkHttpClient(okHttpClient)
            .build()
            .apply {
                when (requestType) {
                    API_REQUEST_JSON_ARRAY -> getAsJSONArray(object : JSONArrayRequestListener {
                        override fun onResponse(response: JSONArray) {
                            jsonArraySuccessListener(response)
                        }

                        override fun onError(error: ANError) {
                            errorHandler(error)
                        }
                    })

                    API_REQUEST_JSON -> getAsJSONObject(object : JSONObjectRequestListener {
                        override fun onResponse(response: JSONObject) {
                            jsonSuccessListener(response)
                        }

                        override fun onError(error: ANError) {
                            errorHandler(error)
                        }
                    })

                    else -> getAsString(object : StringRequestListener {
                        override fun onResponse(response: String) {
                            stringSuccessListener(response)
                        }

                        override fun onError(error: ANError) {
                            errorHandler(error)
                        }
                    })
                }
            }
    }
}

