package com.appspiriment.androidutils

import android.app.Activity
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.os.Message
import android.text.Html
import android.view.View
import android.view.inputmethod.InputMethodManager
import android.widget.TextView
import android.widget.Toast
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AlertDialog

/********************************************
 * Class for methods related to Messages
 * *******************************************/
/**
 * **************************************
 * Method actionTo show an Alert Dialog
 * ****************************************
 */
fun Context.showToast(
    message: String? = null,
    isLong: Boolean = false,
    messageRes: Int? = null
) {
    Toast.makeText(
        this,
        message ?: messageRes?.let { this.getString(it) } ?: "No message info available!",
        if (isLong) Toast.LENGTH_LONG else Toast.LENGTH_SHORT
    ).show()
}

/**
 * **************************************
 * Method actionTo show an Alert Dialog
 * ****************************************
 */
fun Context.startActivityByClass(activityClass: Class<out Activity>, extras: Bundle = Bundle()) {
    startActivity(Intent(this.applicationContext, activityClass).apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
        putExtras(extras)
    })
}


/**
 * **************************************
 * Method to hide Soft Keyboard
 * ****************************************
 */
fun Activity.hideKeyboard() {
    try {
        val imm = this.getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager
        var view = this.currentFocus
        if (view == null) {
            view = View(this)
        }
        imm.hideSoftInputFromWindow(view.windowToken, 0)
    } catch (e: Exception) {
    }
}


/**
 * **************************************
 * Method actionTo show an Alert Dialog with onClick Listeners
 * ****************************************
 */
@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
fun Activity.showMsgDialog(
    title: String? = null,
    titleRes: Int? = null,
    message: String? = null,
    messageRes: Int? = null,
    viewRes: Int? = null,
    view: View? = null,
    positiveButtonTxt: String? = null,
    negativeButtonTxt: String? = null,
    neutralButtonTxt: String? = null,
    positiveButtonRes: Int? = null,
    negativeButtonRes: Int? = null,
    neutralButtonRes: Int? = null,
    finishActivityOnOk: Boolean = false,
    finishActivityAffinity: Boolean = false,
    finishActivityOnCancel: Boolean = false,
    isCancellable: Boolean = false,
    positiveClickListen: () -> Unit = {},
    negativeClickListen: () -> Unit = {},
    neutralClickListen: () -> Unit = {},
    genericClickListen: (Int) -> Unit = {},
    onCancelListen: () -> Unit = {},
    iconRes: Int? = null,
    isHtmlMessage: Boolean = false
) {
    getAlertDialogBuilder(
        this,
        title = title,
        message = message,
        titleRes = titleRes,
        messageRes = messageRes,
        viewRes = viewRes,
        view = view,
        positiveButton = positiveButtonTxt,
        negativeButton = negativeButtonTxt,
        neutralButton = neutralButtonTxt,
        positiveButtonRes = positiveButtonRes,
        negativeButtonRes = negativeButtonRes,
        neutralButtonRes = neutralButtonRes,
        finishActivityOnOk = finishActivityOnOk,
        finishActivityAffinity = finishActivityAffinity,
        finishActivityOnCancel = finishActivityOnCancel,
        isCancellable = isCancellable,
        positiveClickListen = positiveClickListen,
        negativeClickListen = negativeClickListen,
        neutralClickListen = neutralClickListen,
        genericClickListen = genericClickListen,
        onCancelListen = onCancelListen,
        iconRes = iconRes,
        isHtmlMessage = isHtmlMessage
    ).create().show()
}

/**
 * **************************************
 * Method actionTo show an Alert Dialog with onClick Listeners
 * ****************************************
 */
fun Activity.showChoiceDialog(
    optionsList: List<String>? = null,
    optionsRes: Int? = null,
    promptMsg: String? = null,
    promptMsgResId: Int? = null,
    repeatPromptMsg: String? = null,
    repeatPromptMsgResId: Int? = null,
    positiveButtonTxt: String? = null,
    negativeButtonTxt: String? = null,
    neutralButtonTxt: String? = null,
    positiveButtonRes: Int? = null,
    negativeButtonRes: Int? = null,
    neutralButtonRes: Int? = null,
    positiveClickListen: (selectedItems: List<String>) -> Unit = {},
    negativeClickListen: (selectedItems: List<String>, showChoiceDialogLambda: () -> Unit) -> Unit = {_,_ ->},
    neutralClickListen: (selectedItems: List<String>, showChoiceDialogLambda: () -> Unit) -> Unit = {_,_ ->},
    genericClickListen: (which: Int, selectedItems: List<String>, showChoiceDialogLambda: () -> Unit) -> Unit = { _, _, _ -> },
    choiceClickListener: (String) -> Unit = {},
    isMultiple: Boolean = true,
    isOptional: Boolean = true,
    iconRes: Int? = null,
    isRepeat: Boolean = false
) {
    val options = optionsList?.toTypedArray<CharSequence>() ?: optionsRes?.let {
        resources.getStringArray(optionsRes)
    } ?: throw Exception("Should provide either options or options resId")

    val prompt = if (isRepeat)
        repeatPromptMsg ?: repeatPromptMsgResId?.let { getString(it) }
        ?: throw Exception("RptPromptMsg & RptPrompt MsgResId both null")
    else promptMsg ?: promptMsgResId?.let { getString(it) }
    ?: throw Exception("PromptMsg & Prompt MsgResId both null")

    val selectedItem = arrayListOf<String>()
    val builder = getAlertDialogBuilder(
        activity = this,
        title = prompt,
        positiveButton = positiveButtonTxt,
        positiveButtonRes = positiveButtonRes,
        iconRes = iconRes
    ).apply {
        if (isMultiple) {
            setMultiChoiceItems(
                options,
                null
            ) { _: DialogInterface, itemPos: Int, isChecked: Boolean ->
                "${options[itemPos]}".let {
                    selectedItem.run {
                        if (isChecked) add(it) else remove(it)
                    }
                }
            }
        } else {
            setSingleChoiceItems(options, -1) { _: DialogInterface, itemPos: Int ->
                "${options[itemPos]}".let {
                    selectedItem.add(it)
                    choiceClickListener(it)
                }
            }
        }
    }

    val showChoiceDialogLambda = {
        this.showChoiceDialog(
            optionsList,
            optionsRes,
            promptMsg,
            promptMsgResId,
            repeatPromptMsg,
            repeatPromptMsgResId,
            positiveButtonTxt,
            negativeButtonTxt,
            neutralButtonTxt,
            positiveButtonRes,
            negativeButtonRes,
            neutralButtonRes,
            positiveClickListen,
            negativeClickListen,
            neutralClickListen,
            genericClickListen,
            choiceClickListener,
            isMultiple,
            isOptional,
            iconRes,
            true
        )
    }

    DialogInterface.OnClickListener { dialog, which ->
        dialog.dismiss()
        selectedItem.distinct().toList().let {
            genericClickListen(which, it, showChoiceDialogLambda)
            when (which) {
                DialogInterface.BUTTON_POSITIVE -> {
                    if (!isOptional && selectedItem.isEmpty()) {
                        showChoiceDialogLambda.invoke()
                    } else positiveClickListen(it)
                }
                DialogInterface.BUTTON_NEGATIVE -> negativeClickListen(it, showChoiceDialogLambda)
                DialogInterface.BUTTON_NEUTRAL -> neutralClickListen(it, showChoiceDialogLambda)
            }
        }
    }.let {
        getBuilderWithDialogButtons(
            builder = builder,
            activity = this,
            positiveButtonTxt = positiveButtonTxt,
            negativeButtonTxt = negativeButtonTxt,
            neutralButtonTxt = neutralButtonTxt,
            positiveButtonRes = positiveButtonRes,
            negativeButtonRes = negativeButtonRes,
            neutralButtonRes = neutralButtonRes,
            clickListner = it
        ).apply {
            create().show()
        }
    }
}

/**
 * **************************************
 * Method actionTo show an Alert Dialog with onClick Listeners
 * ****************************************
 */
private fun getAlertDialogBuilder(
    activity: Activity,
    title: String? = null,
    message: String? = null,
    titleRes: Int? = null,
    messageRes: Int? = null,
    viewRes: Int? = null,
    view: View? = null,
    positiveButton: String? = null,
    negativeButton: String? = null,
    neutralButton: String? = null,
    positiveButtonRes: Int? = null,
    negativeButtonRes: Int? = null,
    neutralButtonRes: Int? = null,
    finishActivityOnOk: Boolean = false,
    finishActivityAffinity: Boolean = false,
    finishActivityOnCancel: Boolean = false,
    isCancellable: Boolean = false,
    positiveClickListen: () -> Unit = {},
    negativeClickListen: () -> Unit = {},
    neutralClickListen: () -> Unit = {},
    genericClickListen: (Int) -> Unit = {},
    onCancelListen: () -> Unit = {},
    iconRes: Int? = null,
    isHtmlMessage: Boolean = false
): AlertDialog.Builder {

    val dialogView = view ?: viewRes?.let { activity.layoutInflater.inflate(it, null) }

    val titleText = title ?: titleRes?.let { activity.resources.getString(it) }
    ?: throw  Exception("Either Title or Title ResId have to be given!")

    val messageText = message ?: messageRes?.let { activity.resources.getString(it) }

    val onClickListener = DialogInterface.OnClickListener { dialog, which ->
        dialog.dismiss()
        genericClickListen(which)
        when (which) {
            DialogInterface.BUTTON_POSITIVE -> {
                positiveClickListen()
                if (finishActivityOnOk) {
                    if (finishActivityAffinity) activity.finishAffinity() else activity.finish()
                }
            }
            DialogInterface.BUTTON_NEGATIVE -> {
                negativeClickListen()
                if (finishActivityOnCancel)
                    if (finishActivityAffinity) activity.finishAffinity() else activity.finish()
            }
            DialogInterface.BUTTON_NEUTRAL -> neutralClickListen()
        }
    }

    val onCancelListener = DialogInterface.OnCancelListener { dialog ->
        dialog.dismiss()
        onCancelListen()
        if (finishActivityOnCancel)
            if (finishActivityAffinity) activity.finishAffinity() else activity.finish()
    }

    val builder = AlertDialog.Builder(activity).apply {
        setTitle(titleText)
        messageText?.let {
            setMessage(
                if (isHtmlMessage) Html.fromHtml(it) else it
            )
        }
        setCancelable(isCancellable)
        setPositiveButton(positiveButton, onClickListener)
        iconRes?.let { setIcon(it) }

        dialogView?.let { setView(it) }
        setOnCancelListener(onCancelListener)
    }

    return getBuilderWithDialogButtons(
        builder,
        activity,
        positiveButton,
        negativeButton,
        neutralButton,
        positiveButtonRes,
        negativeButtonRes,
        neutralButtonRes,
        onClickListener
    )
}

/**
 * **************************************
 * Method actionTo show an Alert Dialog with onClick Listeners
 * ****************************************
 */
private fun getBuilderWithDialogButtons(
    builder: AlertDialog.Builder,
    activity: Activity,
    positiveButtonTxt: String? = null,
    negativeButtonTxt: String? = null,
    neutralButtonTxt: String? = null,
    positiveButtonRes: Int? = null,
    negativeButtonRes: Int? = null,
    neutralButtonRes: Int? = null,
    clickListner: DialogInterface.OnClickListener
): AlertDialog.Builder {
    return builder.apply {

        positiveButtonTxt?.let {
            setPositiveButton(it, clickListner)
        } ?: positiveButtonRes?.let { posButtonId ->
            setPositiveButton(activity.getString(posButtonId), clickListner)
        } ?: throw Exception("Should provide a positive button")

        negativeButtonTxt?.let {
            setNegativeButton(it, clickListner)
        } ?: negativeButtonRes?.let { negativeButtonId ->
            setNegativeButton(activity.getString(negativeButtonId), clickListner)
        }

        neutralButtonTxt?.let {
            setNeutralButton(it, clickListner)
        } ?: neutralButtonRes?.let { neutralButtonId ->
            setNeutralButton(activity.getString(neutralButtonId), clickListner)
        }
    }
}
