package com.applovin.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;

import com.applovin.impl.privacy.consentFlow.ConsentFlowManager;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.DataCollector;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.health.HealthEvent;
import com.applovin.impl.sdk.health.HealthEventExtraParameterKey;
import com.applovin.impl.sdk.utils.CollectionUtils;
import com.applovin.impl.sdk.utils.JsonUtils;
import com.applovin.impl.sdk.utils.SdkExtraParameterKey;
import com.applovin.impl.sdk.utils.StringUtils;
import com.applovin.impl.sdk.utils.Utils;

import org.json.JSONObject;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RawRes;
import lombok.val;

import static com.applovin.impl.sdk.utils.Utils.kilobytesToBytes;

/**
 * This class contains mutable settings for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public class AppLovinSdkSettings
{
    private static final String TAG = "AppLovinSdkSettings";

    private boolean isVerboseLoggingEnabled;
    private boolean muted;
    private boolean creativeDebuggerEnabled;
    private boolean failAdDisplayIfDontKeepActivitiesIsEnabled = true;

    // Values to set once SDK is attached
    @Nullable
    private String testModeNetworkToSet;
    @Nullable
    private String userIdentifierToSet;
    @Nullable
    private String packageNameOverrideToSet;

    private final Map<String, Object>                       localSettings                  = new HashMap<>(); // NOTE: do not rename `localSettings` - it is used internally via reflection.
    private final Map<String, String>                       extraParameters                = Collections.synchronizedMap( new HashMap<>() );
    private final Map<String, String>                       preInitExtraParametersToReport = Collections.synchronizedMap( new HashMap<>() );
    private final AppLovinTermsAndPrivacyPolicyFlowSettings termsAndPrivacyPolicyFlowSettings;

    @Nullable
    private CoreSdk sdk;
    private String  packageName = "";

    /**
     * Creates an instance of AppLovin SDK's settings object with the given context to extract.
     */
    AppLovinSdkSettings(final Context context)
    {
        this.creativeDebuggerEnabled = true;

        if ( context == null )
        {
            Logger.userError( TAG, "context cannot be null. Please provide a valid context." );
        }

        // We will use the application context in the case the passed context is null or will throw NPEs when accessed.
        val contextToUse = Utils.getValidContext( context );
        this.isVerboseLoggingEnabled = Utils.isVerboseLoggingEnabled( contextToUse );
        this.termsAndPrivacyPolicyFlowSettings = ConsentFlowManager.generateConsentFlowSettingsFromRawResource( contextToUse );
        this.packageName = contextToUse.getPackageName();
        updateExtraParametersFromRawResources( contextToUse );
    }

    /**
     * Get the MAX Terms and Privacy Policy Flow settings.
     */
    public AppLovinTermsAndPrivacyPolicyFlowSettings getTermsAndPrivacyPolicyFlowSettings()
    {
        return termsAndPrivacyPolicyFlowSettings;
    }

    /**
     * Toggle verbose logging of AppLovin SDK. If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     *
     * @param isVerboseLoggingEnabled True if log messages should be output.
     */
    public void setVerboseLogging(boolean isVerboseLoggingEnabled)
    {
        Logger.logApiCall( TAG, "setVerboseLogging(isVerboseLoggingEnabled=" + isVerboseLoggingEnabled + ")" );

        // If enabled from Android manifest, ignore programmatic setting.
        // This makes life easier for PubOps folks when mediation networks override this setting.
        if ( Utils.isVerboseLoggingConfigured() )
        {
            Logger.userError( TAG, "Ignoring setting of verbose logging - it is configured from Android manifest already." );

            if ( Utils.isVerboseLoggingEnabled( null ) != isVerboseLoggingEnabled )
            {
                Logger.userError( TAG, "Attempted to programmatically set verbose logging flag to value different from value configured in Android Manifest." );
            }
        }
        else
        {
            this.isVerboseLoggingEnabled = isVerboseLoggingEnabled;
        }
    }

    /**
     * Check if verbose logging is enabled for the AppLovin SDK.
     * <p>
     * If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     */
    public boolean isVerboseLoggingEnabled()
    {
        return isVerboseLoggingEnabled;
    }

    /**
     * Whether video ads begin in a muted state or not. Defaults to {@code false}.
     */
    public boolean isMuted()
    {
        return muted;
    }

    /**
     * Set whether to begin video ads in a muted state or not.
     *
     * @param muted If ads should begin in a muted state.
     */
    public void setMuted(boolean muted)
    {
        Logger.logApiCall( TAG, "setMuted(muted=" + muted + ")" );

        this.muted = muted;
    }

    /**
     * Set whether the Creative Debugger will be displayed after flipping the device screen down twice. Defaults to {@code true}.
     */
    public void setCreativeDebuggerEnabled(boolean creativeDebuggerEnabled)
    {
        Logger.logApiCall( TAG, "setCreativeDebuggerEnabled(creativeDebuggerEnabled=" + creativeDebuggerEnabled + ")" );

        if ( this.creativeDebuggerEnabled == creativeDebuggerEnabled ) return;

        this.creativeDebuggerEnabled = creativeDebuggerEnabled;

        if ( sdk == null ) return;

        if ( creativeDebuggerEnabled )
        {
            sdk.getCreativeDebuggerService().maybePrepareCreativeDebugger();
        }
        else
        {
            sdk.getCreativeDebuggerService().maybeDisableCreativeDebugger();
        }
    }

    /**
     * Whether the Creative Debugger will be displayed after flipping the device screen down twice. Defaults to {@code true}.
     */
    public boolean isCreativeDebuggerEnabled()
    {
        return creativeDebuggerEnabled;
    }

    /**
     * An identifier for the current user. This identifier will be tied to SDK events and our optional S2S postbacks.
     * <p>
     * If you're using reward validation, you can optionally set an identifier to be included with currency validation postbacks.
     * For example, a username or email. We'll include this in the postback when we ping your currency endpoint from our server.
     */
    @Nullable
    public String getUserIdentifier()
    {
        if ( sdk == null ) return userIdentifierToSet;

        return sdk.getUserTokenManager().getUserId();
    }

    /**
     * Set an identifier for the current user. This identifier will be tied to SDK events and our optional S2S postbacks.
     * <p>
     * If you're using reward validation, you can optionally set an identifier to be included with currency validation postbacks.
     * For example, a username or email. We'll include this in the postback when we ping your currency endpoint from our server.
     *
     * @param userIdentifier The user identifier to be set.
     */
    public void setUserIdentifier(String userIdentifier)
    {
        // Log for better publisher debugging, only in verbose logging in case the `userIdentifier` is a GAID.
        Logger.logApiCall( TAG, "setUserIdentifier(userIdentifier=" + userIdentifier + ")" );

        // MoPub's SDK logs a warning if over 8KB
        if ( StringUtils.isValidString( userIdentifier ) && userIdentifier.length() > kilobytesToBytes( 8 ) )
        {
            Logger.userError( Logger.SDK_TAG, "Provided user id longer than supported (" + userIdentifier.length() + " bytes, " + kilobytesToBytes( 8 ) + " maximum)" );
        }

        if ( sdk != null )
        {
            // Log for better publisher debugging, only in verbose logging in case the `userIdentifier` is a GAID.
            sdk.getLogger().d( TAG, "Setting user id: " + userIdentifier );

            sdk.getUserTokenManager().setUserId( userIdentifier );
        }
        else
        {
            userIdentifierToSet = userIdentifier;
        }
    }

    /**
     * A copy of the extra parameters that are currently set.
     */
    public Map<String, String> getExtraParameters()
    {
        synchronized ( extraParameters )
        {
            // Note: Returning a copy of the extra parameters so that our copy may not be modified by the publisher.
            return CollectionUtils.map( extraParameters );
        }
    }

    /**
     * Set an extra parameter to pass to the AppLovin server.
     *
     * @param key   Parameter key. Must not be null.
     * @param value Parameter value. May be null.
     */
    public void setExtraParameter(final String key, @Nullable final String value)
    {
        Logger.logApiCall( TAG, "setExtraParameter(key=" + key + ", value=" + value + ")" );

        if ( TextUtils.isEmpty( key ) )
        {
            Logger.userError( TAG, "Failed to set extra parameter for null or empty key: " + key );
            return;
        }

        String sanitizedValue = ( value != null ) ? value.trim() : null;

        if ( "test_mode_network".equalsIgnoreCase( key ) )
        {
            if ( sdk != null )
            {
                if ( StringUtils.isValidString( sanitizedValue ) )
                {
                    sdk.getTestModeService().setNetworks( Arrays.asList( sanitizedValue.split( "," ) ) );
                }
                else
                {
                    sdk.getTestModeService().setNetwork( null );
                }
            }
            else
            {
                testModeNetworkToSet = sanitizedValue;
            }
        }
        else if ( SdkExtraParameterKey.FILTER_AD_NETWORK.equals( key ) || SdkExtraParameterKey.ENABLE_SEQUENTIAL_CACHING.equals( key ) )
        {
            // Temp hack for Unity to test - their test apps' package names start with com.unity.
            if ( !packageName.startsWith( "com.unity." ) ) return;
        }
        else if ( SdkExtraParameterKey.DISABLE_ALL_LOGS.equals( key ) )
        {
            // Optimization - just update Logger directly
            Logger.setDisableAllLogs( Boolean.parseBoolean( sanitizedValue ) );
        }
        else if ( SdkExtraParameterKey.PACKAGE_NAME_OVERRIDE.equals( key ) )
        {
            if ( sdk != null )
            {
                // Optimization - just update DataCollector directly
                DataCollector.overridePackageNameIfNeeded( sanitizedValue, sdk );
            }
            else
            {
                packageNameOverrideToSet = sanitizedValue;
            }
        }

        if ( sdk != null )
        {
            val healthEventExtraParameters = CollectionUtils.hashMap( HealthEventExtraParameterKey.DETAILS, key + ":" + value );
            sdk.getUserSessionHealthEventsReporter().submitReport( HealthEvent.SDK_EXTRA_PARAMETER_SET, "postInitExtraParameter", healthEventExtraParameters );
        }
        else
        {
            preInitExtraParametersToReport.put( key, sanitizedValue );
        }

        extraParameters.put( key, sanitizedValue );
    }

    /**
     * The SDK will automatically fail fullscreen ad display and invoke the {@code com.applovin.mediation.MaxAdListener#onAdDisplayFailed(...)}
     * when the "Don't Keep Activities" developer setting is enabled. This setting allows bypassing that.
     * <p>
     * This flag will only be honored in debuggable builds.
     *
     * @param shouldFailAdDisplayIfDontKeepActivitiesIsEnabled Set to @{code false} to disable the SDK automatically failing fullscreen ad display when the "Don't Keep Activities" developer setting is enabled.
     */
    public void setShouldFailAdDisplayIfDontKeepActivitiesIsEnabled(final boolean shouldFailAdDisplayIfDontKeepActivitiesIsEnabled)
    {
        Logger.logApiCall( TAG, "setShouldFailAdDisplayIfDontKeepActivitiesIsEnabled(shouldFailAdDisplayIfDontKeepActivitiesIsEnabled="
                + shouldFailAdDisplayIfDontKeepActivitiesIsEnabled + ")" );

        this.failAdDisplayIfDontKeepActivitiesIsEnabled = shouldFailAdDisplayIfDontKeepActivitiesIsEnabled;
    }

    /**
     * Whether or not the SDK will automatically fail fullscreen ad display when the "Don't Keep Activities" developer setting is enabled. Defaults to {@code true}.
     */
    public boolean shouldFailAdDisplayIfDontKeepActivitiesIsEnabled()
    {
        return failAdDisplayIfDontKeepActivitiesIsEnabled;
    }

    public void attachAppLovinSdk(CoreSdk sdk)
    {
        this.sdk = sdk;

        //
        // Set any values that depend on an SDK instance
        //

        if ( StringUtils.isValidString( testModeNetworkToSet ) )
        {
            sdk.getTestModeService().setNetworks( Arrays.asList( testModeNetworkToSet.split( "," ) ) );

            testModeNetworkToSet = null;
        }

        if ( userIdentifierToSet != null )
        {
            // Log for better publisher debugging, only in verbose logging in case the `userIdentifier` is a GAID.
            sdk.getLogger().d( TAG, "Setting user id: " + userIdentifierToSet );

            sdk.getUserTokenManager().setUserId( userIdentifierToSet );
            userIdentifierToSet = null;
        }

        if ( StringUtils.isValidString( packageNameOverrideToSet ) )
        {
            DataCollector.overridePackageNameIfNeeded( packageNameOverrideToSet, sdk );

            packageNameOverrideToSet = null;
        }

        // Track SDK extra parameters that were set before SDK initialization
        for ( val entry : preInitExtraParametersToReport.entrySet() )
        {
            val healthEventExtraParameters = CollectionUtils.hashMap( HealthEventExtraParameterKey.DETAILS, entry.getKey() + ":" + entry.getValue() );
            sdk.getUserSessionHealthEventsReporter().submitReport( HealthEvent.SDK_EXTRA_PARAMETER_SET, "preInitExtraParameter", healthEventExtraParameters );
        }

        preInitExtraParametersToReport.clear();
    }

    @SuppressLint("DiscouragedApi")
    private void updateExtraParametersFromRawResources(final Context context)
    {
        @RawRes val settingsRawJsonResourceId = context.getResources().getIdentifier( ConsentFlowManager.KEY_APPLOVIN_SDK_SETTINGS_RAW_RESOURCE_NAME, "raw", context.getPackageName() );
        if ( settingsRawJsonResourceId == 0 ) return;

        val settingsJsonString = Utils.getRawResourceString( settingsRawJsonResourceId, context, null );
        val settingsJson = StringUtils.isValidString( settingsJsonString ) ? JsonUtils.jsonObjectFromJsonString( settingsJsonString, new JSONObject() ) : new JSONObject();
        val parameters = JsonUtils.tryToStringMap( settingsJson );

        extraParameters.putAll( parameters );
    }

    @Override
    @NonNull
    public String toString()
    {
        return "AppLovinSdkSettings{" +
                "isVerboseLoggingEnabled=" + isVerboseLoggingEnabled +
                ", muted=" + muted +
                ", creativeDebuggerEnabled=" + creativeDebuggerEnabled +
                '}';
    }
}
