package com.applovin.sdk;

import android.content.Context;
import android.net.Uri;

import com.applovin.impl.mediation.MaxMediatedNetworkInfoImpl;
import com.applovin.impl.mediation.utils.MediationUtils;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.utils.CollectionUtils;
import com.applovin.impl.sdk.utils.JsonUtils;
import com.applovin.mediation.MaxMediatedNetworkInfo;
import com.applovin.mediation.MaxSegmentCollection;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import androidx.annotation.Nullable;
import lombok.val;

/**
 * This is the base class for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public final class AppLovinSdk
{
    /**
     * Current SDK version. The returned value will be in the format of "<Major>.<Minor>.<Revision>".
     */
    public static final String VERSION = getVersion();

    public static final int VERSION_CODE = getVersionCode();

    private static final String TAG = "AppLovinSdk";

    private static       AppLovinSdk instance;
    private static final Object      instanceLock = new Object();

    /**
     * Use {@link AppLovinSdk#a()} to access.
     * NOTE: This field used to be private and retrieved via reflection, but was causing crashes.
     *
     * @hide
     */
    private final CoreSdk coreSdk;

    /**
     * Listener interface to be used with {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, SdkInitializationListener)}
     */
    public interface SdkInitializationListener
    {
        void onSdkInitialized(AppLovinSdkConfiguration config);
    }

    //region High Level SDK Properties

    /**
     * Get client SDK key.
     *
     * @return Client SDK key.
     * @hide
     */
    public String getSdkKey()
    {
        return coreSdk.getSdkKey();
    }

    /**
     * Get SDK settings provided on initialization.
     *
     * @return Provided settings.
     */
    public AppLovinSdkSettings getSettings()
    {
        return coreSdk.getSettings();
    }

    /**
     * Get the SDK configuration object provided upon initialization.
     *
     * @return An instance of the SDK configuration.
     */
    public AppLovinSdkConfiguration getConfiguration()
    {
        return coreSdk.getConfiguration();
    }

    /**
     * Get the {@link com.applovin.mediation.MaxSegmentCollection} object that the SDK had been initialized with.
     */
    public MaxSegmentCollection getSegmentCollection()
    {
        return coreSdk.getSegmentCollection();
    }

    //endregion

    //region SDK Services

    /**
     * Get an instance of the AppLovin Ad service. This service is used to fetch ads from AppLovin servers.
     *
     * @return Ad service. Guaranteed not to be null.
     */
    public AppLovinAdService getAdService()
    {
        return coreSdk.getAdService();
    }

    /**
     * Get an instance of the AppLovin event service. This service is used to track post-install user events.
     *
     * @return Event service. Guaranteed not to be null.
     */
    public AppLovinEventService getEventService()
    {
        return coreSdk.getEventService();
    }

    /**
     * The CMP service, which provides direct APIs for interfacing with the Google-certified CMP installed, if any.
     *
     * @return CMP service. Guaranteed not to be {@code null}.
     */
    public AppLovinCmpService getCmpService()
    {
        return coreSdk.getCmpService();
    }

    //endregion

    //region MAX

    /**
     * Returns the list of available mediation networks.
     *
     * @return List of {@link MaxMediatedNetworkInfo} objects.
     */
    public List<MaxMediatedNetworkInfo> getAvailableMediatedNetworks()
    {
        JSONArray availableMediationAdapters = MediationUtils.getAvailableMediationAdapters( coreSdk );
        List<MaxMediatedNetworkInfo> availableMediatedNetworks = new ArrayList<>( availableMediationAdapters.length() );

        for ( int i = 0; i < availableMediationAdapters.length(); i++ )
        {
            JSONObject adapter = JsonUtils.getJSONObject( availableMediationAdapters, i, null );
            availableMediatedNetworks.add( new MaxMediatedNetworkInfoImpl( adapter ) );
        }

        return availableMediatedNetworks;
    }

    /**
     * Present the mediation debugger UI.
     * This debugger tool provides the status of your integration for each third-party ad network.
     * <p>
     * Please call this method after the SDK has initialized, e.g. {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, SdkInitializationListener)}.
     */
    public void showMediationDebugger()
    {
        coreSdk.showMediationDebugger();
    }

    /**
     * Present the mediation debugger UI.
     * This debugger tool provides the status of your integration for each third-party ad network.
     * <p>
     * Please call this method after the SDK has initialized, e.g. {@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, SdkInitializationListener)}.
     *
     * @param amazonAdSizes A map of the MAX Ad Unit Id to Amazon Publisher Services' {@code com.amazon.device.ads.DTBAdSize}.
     */
    public void showMediationDebugger(@Nullable final Map<String, List<?>> amazonAdSizes)
    {
        coreSdk.showMediationDebugger( amazonAdSizes );
    }

    /**
     * Present the Creative Debugger UI.
     * This debugger tool provides information for recently displayed ads.
     */
    public void showCreativeDebugger()
    {
        coreSdk.showCreativeDebugger();
    }

    //endregion

    //region Axon Advertiser Services

    /**
     * Processes the incoming deep link.
     *
     * @param incomingUri The URL used to open or bring the app into the foreground. It may contain query parameters that need to be filtered and processed.
     */
    public void processDeepLink(final Uri incomingUri)
    {
        coreSdk.processDeepLink( incomingUri );
    }

    //endregion

    //region SDK Initialization

    /**
     * Check whether the SDK has fully been initialized without errors and the completion callback called.
     */
    public boolean isInitialized()
    {
        return coreSdk.isEnabled();
    }

    /**
     * @param context Android application context.
     *
     * @return An instance of AppLovinSDK
     */
    public static AppLovinSdk getInstance(final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        synchronized ( instanceLock )
        {
            if ( instance == null )
            {
                val sdk = new CoreSdk( new AppLovinSdkSettings( context ), context );
                val wrapper = new AppLovinSdk( sdk );
                sdk.setWrappingSdk( wrapper );  // The public-facing interface used by the pub

                instance = wrapper;
            }

            return instance;
        }
    }

    /**
     * Initializes the SDK with the given initialization configuration and listener.
     * <p>
     * The callback will be invoked on the main thread.
     *
     * @param initializationConfiguration The configuration to initialize the SDK with.
     * @param listener                    The callback that will be run on the main thread when the SDK finishes initializing.
     */
    public void initialize(final AppLovinSdkInitializationConfiguration initializationConfiguration, @Nullable final SdkInitializationListener listener)
    {
        coreSdk.initialize( initializationConfiguration, listener );
    }

    protected void reinitialize(final Boolean hasUserConsent, final Boolean doNotSell)
    {
        // Re-init if we have already initialized before
        if ( coreSdk.isInitializationConfigured() )
        {
            coreSdk.reinitialize();
        }

        coreSdk.notifyPrivacySettingUpdated();

        // Fire an event if the user consent state has explicitly changed
        if ( hasUserConsent != null )
        {
            coreSdk.getLogger().i( TAG, "Toggled 'huc' to " + hasUserConsent );

            val parameters = CollectionUtils.map( "value", hasUserConsent.toString() );
            getEventService().trackEvent( "huc", parameters );
        }

        // Fire an event if the "do not sell" state has explicitly changed
        if ( doNotSell != null )
        {
            coreSdk.getLogger().i( TAG, "Toggled 'dns' to " + doNotSell );

            val parameters = CollectionUtils.map( "value", doNotSell.toString() );
            getEventService().trackEvent( "dns", parameters );
        }
    }

    //endregion

    //region Private Functions

    /**
     * Get the SDK version string from the Build.gradle file.
     * <p>
     * Abstracted into method to prevent compiler optimization in adapters which hardcodes references to `VERSION` and `VERSION_CODE`
     * when it should dynamically retrieve versioning info depending on current SDK installed.
     */
    private static String getVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    /**
     * Get the SDK version code from the Build.gradle file.
     * <p>
     * Abstracted into method to prevent compiler optimization in adapters which hardcodes references to `VERSION` and `VERSION_CODE`
     * when it should dynamically retrieve versioning info depending on current SDK installed.
     */
    private static int getVersionCode()
    {
        return BuildConfig.VERSION_CODE;
    }

    private AppLovinSdk(CoreSdk sdk)
    {
        this.coreSdk = sdk;
    }

    //endregion

    //region Miscellaneous Public Functions

    @Override
    public String toString()
    {
        return "AppLovinSdk{" +
                "sdkKey='" + getSdkKey() + "'" +
                ", isInitialized=" + isInitialized() +
                ", isFirstSession=" + coreSdk.isFirstSession() +
                '}';
    }

    /**
     * This returns the {@link CoreSdk} instance and should not be used by publishers.
     *
     * @return The {@link CoreSdk} instance.
     * @hide
     */
    public CoreSdk a()
    {
        return coreSdk;
    }

    //endregion
}
