package com.applovin.sdk;

import android.content.Context;

import com.applovin.impl.sdk.AppLovinSdkInitializationConfigurationImpl;
import com.applovin.mediation.MaxSegmentCollection;

import java.util.List;

import androidx.annotation.Nullable;

/**
 * This class contains configurable fields for the initialization of the AppLovin SDK.
 * To be used in @{@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}.
 */
public interface AppLovinSdkInitializationConfiguration
{
    static Builder builder(final String sdkKey, final Context context)
    {
        return new AppLovinSdkInitializationConfigurationImpl.BuilderImpl( sdkKey, context );
    }

    /**
     * @return the SDK key for the AppLovin SDK.
     */
    @Nullable
    String getSdkKey();

    /**
     * @return mediation provider.
     */
    @Nullable
    String getMediationProvider();

    /**
     * @return the plugin version for the mediation adapter or plugin.
     */
    @Nullable
    String getPluginVersion();

    /**
     * @return A readonly property of a {@link com.applovin.mediation.MaxSegmentCollection}, representing a collection of segments.
     */
    @Nullable
    MaxSegmentCollection getSegmentCollection();

    /**
     * @return the list of advertising identifiers that will receive test ads.
     */
    List<String> getTestDeviceAdvertisingIds();

    /**
     * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
     */
    List<String> getAdUnitIds();

    /**
     * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    boolean isExceptionHandlerEnabled();

    /**
     * Builder class used to create an {@link AppLovinSdkInitializationConfiguration} object.
     * This class contains configurable fields for the initialization of the AppLovin SDK.
     */
    interface Builder
    {
        /**
         * Set mediation provider using one of the provided strings above specified by {@link AppLovinMediationProvider}, or your own if not defined.
         */
        Builder setMediationProvider(@Nullable final String mediationProvider);

        /**
         * Sets the plugin version for the mediation adapter or plugin.
         */
        Builder setPluginVersion(@Nullable final String version);

        /**
         * Set the {@link MaxSegmentCollection}, a collection of segments for this instance of the SDK.
         */
        Builder setSegmentCollection(final MaxSegmentCollection segmentCollection);

        /**
         * Enable devices to receive test ads, by passing in the advertising identifier (GAID or App Set ID) of each test device.
         * Refer to AppLovin logs for the GAID or App Set ID of your current device.
         */
        Builder setTestDeviceAdvertisingIds(final List<String> testDeviceAdvertisingIds);

        /**
         * Set the MAX ad unit ids that will be used for this instance of the SDK. 3rd-party SDKs will be initialized with the credentials configured for these ad unit ids.
         */
        Builder setAdUnitIds(final List<String> adUnitIds);

        /**
         * Set whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        Builder setExceptionHandlerEnabled(final boolean exceptionHandlerEnabled);

        /**
         * Build method to create an instance of {@link AppLovinSdkInitializationConfiguration}.
         *
         * @return an instance of {@link AppLovinSdkInitializationConfiguration}.
         */
        AppLovinSdkInitializationConfiguration build();

        /**
         * @return the SDK key for the AppLovin SDK.
         */
        String getSdkKey();

        /**
         * @return mediation provider.
         */
        @Nullable
        String getMediationProvider();

        /**
         * @return the plugin version for the mediation adapter or plugin.
         */
        @Nullable
        String getPluginVersion();

        /**
         * @return the {@link MaxSegmentCollection} object.
         */
        @Nullable
        MaxSegmentCollection getSegmentCollection();

        /**
         * @return the list of advertising identifiers that will receive test ads.
         */
        List<String> getTestDeviceAdvertisingIds();

        /**
         * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
         */
        List<String> getAdUnitIds();

        /**
         * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        boolean isExceptionHandlerEnabled();
    }
}
