package com.applovin.sdk;

import com.applovin.impl.sdk.AppLovinTargetingDataImpl;

import java.util.List;

import androidx.annotation.Nullable;

/**
 * This class allows you to provide user or app data that will improve how we target ads.
 *
 * @author Basil Shikin
 */
public interface AppLovinTargetingData
{
    /**
     * This enumeration represents content ratings for the ads shown to users.
     * They correspond to IQG Media Ratings.
     */
    enum AdContentRating
    {
        NONE,
        ALL_AUDIENCES,
        EVERYONE_OVER_TWELVE,
        MATURE_AUDIENCES
    }

    /**
     * This enumeration represents gender.
     */
    enum Gender
    {
        UNKNOWN,
        FEMALE,
        MALE,
        OTHER
    }

    /**
     * @return builder to create an {@link AppLovinTargetingData} object.
     */
    static Builder builder()
    {
        return new AppLovinTargetingDataImpl.BuilderImpl();
    }

    /**
     * @return the gender of user.
     */
    @Nullable
    Gender getGender();

    /**
     * @return the maximum ad content rating shown the user.
     */
    @Nullable
    AdContentRating getMaximumAdContentRating();

    /**
     * @return the year of birth of the user.
     */
    @Nullable
    Integer getYearOfBirth();

    /**
     * @return the email of the user.
     */
    @Nullable
    String getEmail();

    /**
     * @return the phone number of the user.
     */
    @Nullable
    String getPhoneNumber();

    /**
     * @return keywords describing the application.
     */
    @Nullable
    List<String> getKeywords();

    /**
     * @return the interests of the user.
     */
    @Nullable
    List<String> getInterests();

    //region AppLovinTargetingData.Builder

    /**
     * Builder class used to create an {@link AppLovinTargetingData} object.
     */
    interface Builder
    {
        /**
         * Set the gender of user.
         * Set this property to {@link Gender#UNKNOWN} or {@code null} to clear this value.
         */
        Builder setGender(@Nullable final Gender gender);

        /**
         * Set the maximum ad content rating shown the user.
         * Set this property to {@link AdContentRating#NONE} or {@code null} to clear this value.
         */
        Builder setMaximumAdContentRating(@Nullable final AdContentRating maximumAdContentRating);

        /**
         * Set the year of birth of the user.
         * Set this property to {@code null} to clear this value.
         */
        Builder setYearOfBirth(@Nullable final Integer yearOfBirth);

        /**
         * Set the email of the user.
         * Set this property to {@code null} to clear this value.
         */
        Builder setEmail(@Nullable final String email);

        /**
         * Set the phone number of the user. Do not include the country calling code.
         * Set this property to {@code null} to clear this value.
         */
        Builder setPhoneNumber(@Nullable final String phoneNumber);

        /**
         * Set the keywords describing the application.
         * Set this property to {@code null} to clear this value.
         */
        Builder setKeywords(@Nullable final List<String> keywords);

        /**
         * Set the interests of the user.
         * Set this property to {@code null} to clear this value.
         */
        Builder setInterests(@Nullable final List<String> interests);

        /**
         * Build method to create an instance of {@link AppLovinTargetingData}.
         *
         * @return an instance of {@link AppLovinTargetingData}.
         */
        AppLovinTargetingData build();

        /**
         * @return the gender of user.
         */
        @Nullable
        Gender getGender();

        /**
         * @return the maximum ad content rating shown the user.
         */
        @Nullable
        AdContentRating getMaximumAdContentRating();

        /**
         * @return the year of birth of the user.
         */
        @Nullable
        Integer getYearOfBirth();

        /**
         * @return the email of the user.
         */
        @Nullable
        String getEmail();

        /**
         * @return the phone number of the user.
         */
        @Nullable
        String getPhoneNumber();

        /**
         * @return keywords describing the application.
         */
        @Nullable
        List<String> getKeywords();

        /**
         * @return the interests of the user.
         */
        @Nullable
        List<String> getInterests();
    }

    //endregion

    //region Deprecated APIs

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setYearOfBirth(Integer)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setYearOfBirth(@Nullable final Integer yearOfBirth);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setGender(Gender)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setGender(@Nullable final Gender gender);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setMaximumAdContentRating(AdContentRating)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setMaximumAdContentRating(@Nullable final AdContentRating maximumAdContentRating);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setEmail(String)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setEmail(@Nullable final String email);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setPhoneNumber(String)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setPhoneNumber(@Nullable final String phoneNumber);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setKeywords(List)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setKeywords(@Nullable final List<String> keywords);

    /**
     * @deprecated This setter is deprecated and will be removed in a future SDK version. Please use {@link AppLovinTargetingData.Builder#setInterests(List)} and building {@link AppLovinTargetingData#builder()} in {@link AppLovinSdkInitializationConfiguration.Builder#setTargetingData(AppLovinTargetingData)}.
     */
    @Deprecated
    void setInterests(@Nullable final List<String> interests);

    /**
     * @deprecated This function is deprecated and will be removed in a future SDK version.
     */
    @Deprecated
    void clearAll();

    //endregion
}

