package com.applovin.mediation.ads;

import android.app.Activity;
import android.content.Context;
import android.text.TextUtils;
import android.view.ViewGroup;

import com.applovin.impl.mediation.ads.MaxAdImplBase;
import com.applovin.impl.mediation.ads.MaxFullscreenAdImpl;
import com.applovin.mediation.MaxAdExpirationListener;
import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxAdRequestListener;
import com.applovin.mediation.MaxAdRevenueListener;
import com.applovin.mediation.MaxAdReviewListener;
import com.applovin.mediation.MaxRewardedAdListener;
import com.applovin.sdk.AppLovinSdk;

import java.lang.ref.WeakReference;

import androidx.annotation.Nullable;
import androidx.lifecycle.Lifecycle;

/**
 * Created by Thomas So on June 01 2020
 * <p>
 * This class represents a fullscreen ad which the user can skip and be granted a reward upon successful completion of the ad.
 */
public class MaxRewardedInterstitialAd
        implements MaxFullscreenAdImpl.ActivityProvider
{
    private static final String TAG = "MaxRewardedInterstitialAd";

    private static WeakReference<Activity> sActivityRef = new WeakReference<>( null );

    private final MaxFullscreenAdImpl mImplementation;

    /**
     * Create a new MAX rewarded interstitial.
     *
     * @param adUnitId Ad unit id to load ads for. Must not be null.
     * @param context  Context to be used. Must not be null.
     */
    public MaxRewardedInterstitialAd(final String adUnitId, final Context context)
    {
        this( adUnitId, AppLovinSdk.getInstance( context ), context );
    }

    /**
     * Create a new MAX rewarded interstitial.
     *
     * @param adUnitId Ad unit id to load ads for. Must not be null.
     * @param sdk      SDK to use. Must not be null. An instance of the SDK may be obtained by calling <code>AppLovinSdk.getInstance()</code>.
     * @param context  Context to be used. Must not be null.
     */
    public MaxRewardedInterstitialAd(final String adUnitId, final AppLovinSdk sdk, final Context context)
    {
        MaxAdImplBase.logApiCall( TAG, "MaxRewardedInterstitialAd(adUnitId=" + adUnitId + ", sdk=" + sdk + ", context=" + context + ")" );

        // Check input
        if ( adUnitId == null ) throw new IllegalArgumentException( "No ad unit ID specified" );
        if ( TextUtils.isEmpty( adUnitId ) ) throw new IllegalArgumentException( "Empty ad unit ID specified" );
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );

        //TODO: Remove this when we remove Activity based API's
        if ( context instanceof Activity )
        {
            sActivityRef = new WeakReference<>( (Activity) context );
        }

        mImplementation = new MaxFullscreenAdImpl( adUnitId, MaxAdFormat.REWARDED_INTERSTITIAL, this, "MaxRewardedInterstitialAd", sdk.a(), context );
    }

    /**
     * Set a listener that will be notified about ad events.
     *
     * @param listener Listener to be notified. May be null.
     */
    public void setListener(final MaxRewardedAdListener listener)
    {
        mImplementation.logApiCall( "setListener(listener=" + listener + ")" );
        mImplementation.setListener( listener );
    }

    /**
     * Set a listener that will be notified about ad revenue events.
     *
     * @param listener Listener to be notified. May be null.
     */
    public void setRevenueListener(final MaxAdRevenueListener listener)
    {
        mImplementation.logApiCall( "setRevenueListener(listener=" + listener + ")" );
        mImplementation.setRevenueListener( listener );
    }

    /**
     * Set a listener that will be notified about ad request events.
     *
     * @param listener Listener to be notified. May be null.
     */
    public void setRequestListener(final MaxAdRequestListener listener)
    {
        mImplementation.logApiCall( "setRequestListener(listener=" + listener + ")" );
        mImplementation.setRequestListener( listener );
    }

    /**
     * Set a listener that will be notified about ad expiration events.
     *
     * @param listener Listener to be notified. May be {@code null}.
     */
    public void setExpirationListener(@Nullable final MaxAdExpirationListener listener)
    {
        mImplementation.logApiCall( "setExpirationListener(listener=" + listener + ")" );
        mImplementation.setExpirationListener( listener );
    }

    /**
     * Set a listener that will be notified when the Ad Review creative id is available.
     */
    public void setAdReviewListener(final MaxAdReviewListener listener)
    {
        mImplementation.logApiCall( "setAdReviewListener(listener=" + listener + ")" );
        mImplementation.setAdReviewListener( listener );
    }

    /**
     * Load ad for the current rewarded interstitial.
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about ad load state.
     */
    public void loadAd()
    {
        mImplementation.logApiCall( "loadAd()" );
        mImplementation.loadAd();
    }

    /**
     * Show the loaded rewarded interstitial ad.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param activity Activity to show the ad in. Must not be null.
     */
    public void showAd(final Activity activity)
    {
        showAd( null, activity );
    }

    /**
     * Show the loaded rewarded interstitial ad for a given placement to tie ad events to.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param placement The placement to tie the showing ad's events to.
     * @param activity  Activity to show the ad in. Must not be null.
     */
    public void showAd(final String placement, final Activity activity)
    {
        showAd( placement, null, activity );
    }

    /**
     * Show the loaded rewarded interstitial ad for a given placement and custom data to tie ad events to.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param placement  The placement to tie the showing ad's events to.
     * @param customData The custom data to tie the showing ad's events to. Maximum size is 8KB.
     * @param activity   Activity to show the ad in, Must not be null.
     */
    public void showAd(final String placement, final String customData, final Activity activity)
    {
        mImplementation.logApiCall( "showAd(placement=" + placement + ", customData=" + customData + ", activity=" + activity + ")" );
        mImplementation.showAd( placement, customData, activity );
    }

    /**
     * Show the loaded rewarded interstitial ad in a container view.
     * <p>
     * Note: This {@link MaxRewardedInterstitialAd} ad can only show ads in container views after calling this method.
     * Please create another instance of {@link MaxRewardedInterstitialAd} if you wish to show ads in a fullscreen activity as well.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param containerView ViewGroup used to show the ad in.
     * @param lifecycle     Lifecycle object to manage ad lifecycle events in container views.
     * @param activity      Activity that shows the container view. Must not be null.
     */
    public void showAd(final ViewGroup containerView, final Lifecycle lifecycle, final Activity activity)
    {
        showAd( null, containerView, lifecycle, activity );
    }

    /**
     * Show the loaded rewarded interstitial ad for a given placement to tie ad events to in a container view.
     * <p>
     * Note: This {@link MaxRewardedInterstitialAd} ad can only show ads in container views after calling this method.
     * Please create another instance of {@link MaxRewardedInterstitialAd} if you wish to show ads in a fullscreen activity as well.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param placement     The placement to tie the showing ad's events to.
     * @param containerView ViewGroup used to show the ad in.
     * @param lifecycle     Lifecycle object to manage ad lifecycle events in container views.
     * @param activity      Activity that shows the container view. Must not be null.
     */
    public void showAd(final String placement, final ViewGroup containerView, final Lifecycle lifecycle, final Activity activity)
    {
        showAd( placement, null, containerView, lifecycle, activity );
    }

    /**
     * Show the loaded rewarded interstitial ad for a given placement and custom data to tie ad events to in a container view.
     * <p>
     * Note: This {@link MaxRewardedInterstitialAd} ad can only show ads in container views after calling this method.
     * Please create another instance of {@link MaxRewardedInterstitialAd} if you wish to show ads in a fullscreen activity as well.
     * <p>
     * Use {@link MaxRewardedInterstitialAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedInterstitialAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param placement     The placement to tie the showing ad's events to.
     * @param customData    The custom data to tie the showing ad's events to. Maximum size is 8KB.
     * @param containerView ViewGroup used to show the ad in.
     * @param lifecycle     Lifecycle object to manage ad lifecycle events in container views.
     * @param activity      Activity that shows the container view. Must not be null.
     */
    public void showAd(final String placement, final String customData, final ViewGroup containerView, final Lifecycle lifecycle, final Activity activity)
    {
        mImplementation.logApiCall( "showAd(placement=" + placement + ", customData=" + customData + ", containerView=" + containerView + ", lifecycle=" + lifecycle + ", activity=" + activity + ")" );
        mImplementation.showAd( placement, customData, containerView, lifecycle, activity );
    }

    /**
     * Check if this ad is ready to be shown.
     *
     * @return <code>true</code> if the ad is ready to be shown.
     */
    public boolean isReady()
    {
        boolean isReady = mImplementation.isReady();
        mImplementation.logApiCall( "isReady() " + isReady + " for ad unit id " + mImplementation.getAdUnitId() );
        return isReady;
    }

    /**
     * Destroy current ad and fully remove it from memory in the next GC cycle.
     */
    public void destroy()
    {
        mImplementation.logApiCall( "destroy()" );
        mImplementation.destroy();
    }

    @Override
    public Activity getActivity()
    {
        mImplementation.logApiCall( "getActivity()" );
        return sActivityRef.get();
    }

    /**
     * Set an extra parameter to pass to the server.
     *
     * @param key   Parameter key. Must not be null.
     * @param value Parameter value. May be null.
     */
    public void setExtraParameter(final String key, final String value)
    {
        mImplementation.logApiCall( "setExtraParameter(key=" + key + ", value=" + value + ")" );
        mImplementation.setExtraParameter( key, value );
    }

    /**
     * Set a local extra parameter to pass to the adapter instances. Will not be available in the {@code MaxAdapter#initialize()} method.
     *
     * @param key   Parameter key. Must not be null.
     * @param value Parameter value. May be null.
     */
    public void setLocalExtraParameter(final String key, final Object value)
    {
        mImplementation.logApiCall( "setLocalExtraParameter(key=" + key + ", value=" + value + ")" );
        mImplementation.setLocalExtraParameter( key, value );
    }

    @Override
    public String toString()
    {
        // Paranoia from past battles with Huawei where implementation may not have been initialized when toString() is called
        return "" + mImplementation;
    }

    /**
     * @deprecated Use {@link #showAd(Activity)} instead.
     */
    @Deprecated
    public void showAd()
    {
        showAd( (String) null );
    }

    /**
     * @deprecated Use {@link #showAd(String, Activity)} instead.
     */
    @Deprecated
    public void showAd(final String placement)
    {
        showAd( placement, (String) null );
    }

    /**
     * @deprecated Use {@link #showAd(String, String, Activity)} instead.
     */
    @Deprecated
    public void showAd(final String placement, final String customData)
    {
        mImplementation.logApiCall( "showAd(placement=" + placement + ", customData=" + customData + ")" );
        mImplementation.showAd( placement, customData, getActivity() );
    }

    /**
     * @deprecated Use {@link #showAd(ViewGroup, Lifecycle, Activity)} instead.
     */
    @Deprecated
    public void showAd(final ViewGroup containerView, final Lifecycle lifecycle)
    {
        showAd( null, containerView, lifecycle );
    }

    /**
     * @deprecated Use {@link #showAd(String, ViewGroup, Lifecycle, Activity)} instead.
     */
    @Deprecated
    public void showAd(final String placement, final ViewGroup containerView, final Lifecycle lifecycle)
    {
        showAd( placement, null, containerView, lifecycle );
    }

    /**
     * @deprecated Use {@link #showAd(String, String, ViewGroup, Lifecycle, Activity)} )} instead.
     */
    @Deprecated
    public void showAd(final String placement, final String customData, final ViewGroup containerView, final Lifecycle lifecycle)
    {
        mImplementation.logApiCall( "showAd(placement=" + placement + ", customData=" + customData + ", containerView=" + containerView + ", lifecycle=" + lifecycle + ")" );
        mImplementation.showAd( placement, customData, containerView, lifecycle, getActivity() );
    }
}
