package com.applovin.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;

import com.applovin.impl.privacy.consentFlow.ConsentFlowManager;
import com.applovin.impl.privacy.consentFlow.ConsentFlowSettings;
import com.applovin.impl.sdk.AppLovinSdkSettingsBase;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.utils.AppLovinSdkExtraParameterKey;
import com.applovin.impl.sdk.utils.CollectionUtils;
import com.applovin.impl.sdk.utils.JsonUtils;
import com.applovin.impl.sdk.utils.StringUtils;
import com.applovin.impl.sdk.utils.Utils;
import com.applovin.sdk.AppLovinSdkConfiguration.ConsentFlowUserGeography;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RawRes;
import lombok.val;

/**
 * This class contains settings for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public class AppLovinSdkSettings
        extends AppLovinSdkSettingsBase
{
    /**
     * This interface contains settings that enable the AppLovin Unified consent Flow.
     */
    public interface TermsAndPrivacyPolicyFlowSettings
            extends TermsFlowSettings
    {
        /**
         * The current debug geography of the user.
         */
        ConsentFlowUserGeography getDebugUserGeography();

        /**
         * Set debug user geography. You may use this to test CMP flow by setting this to {@link ConsentFlowUserGeography#GDPR}.
         * <p>
         * The flow would only be shown to new users. If you wish to test the flow after completing the CMP prompt, you would need to delete and re-install the app.
         * <p>
         * NOTE: The debug geography is used only when the app is in debug mode.
         */
        void setDebugUserGeography(final ConsentFlowUserGeography debugUserGeography);
    }

    /**
     * This interface contains settings that enable the AppLovin Terms Flow.
     */
    public interface TermsFlowSettings
    {
        /**
         * Whether or not the consent flow is currently enabled.
         */
        boolean isEnabled();

        /**
         * Set whether to enable the consent flow or not.
         */
        void setEnabled(final boolean enabled);

        /**
         * The current terms of service URL.
         */
        @Nullable
        Uri getPrivacyPolicyUri();

        /**
         * Set the privacy policy URL.
         */
        void setPrivacyPolicyUri(final Uri privacyPolicyUri);

        /**
         * Get the current terms of service URL.
         */
        @Nullable
        Uri getTermsOfServiceUri();

        /**
         * Set the terms of service URL.
         */
        void setTermsOfServiceUri(final Uri termsOfServiceUri);
    }

    private static final String TAG = "AppLovinSdkSettings";

    private boolean isVerboseLoggingEnabled;
    private boolean muted;
    private boolean creativeDebuggerEnabled;
    private boolean exceptionHandlerEnabled;
    private boolean locationCollectionEnabled;
    private boolean failAdDisplayIfDontKeepActivitiesIsEnabled = true;

    // Values to set once SDK is attached
    private String testModeNetworkToSet;

    private final Map<String, Object> localSettings            = new HashMap<>(); // NOTE: do not rename `localSettings` - it is used internally via reflection.
    private final Map<String, String> metaData                 = new HashMap<>(); // NOTE: on not rename `metaData` - it is used internally via reflection.
    private       List<String>        testDeviceAdvertisingIds = Collections.emptyList();
    private       List<String>        initializationAdUnitIds  = Collections.emptyList();
    private final Map<String, String> extraParameters          = new HashMap<>();
    private final Object              extraParametersLock      = new Object();

    @Nullable
    private CoreSdk sdk;
    private String  packageName = "";

    /**
     * Creates an instance of AppLovin SDK's settings object with the given context to extract.
     */
    public AppLovinSdkSettings(final Context context)
    {
        this.creativeDebuggerEnabled = true;
        this.exceptionHandlerEnabled = true;
        this.locationCollectionEnabled = true;

        if ( context == null )
        {
            Logger.userError( TAG, "context cannot be null. Please provide a valid context." );
        }

        // We will use the application context in the case the passed context is null or will throw NPEs when accessed.
        val contextToUse = Utils.getValidContext( context );
        this.isVerboseLoggingEnabled = Utils.isVerboseLoggingEnabled( contextToUse );
        this.backingConsentFlowSettings = ConsentFlowManager.generateConsentFlowSettingsFromRawResource( contextToUse );
        this.packageName = contextToUse.getPackageName();
        updateExtraParametersFromRawResources( contextToUse );
    }

    /**
     * Get the MAX Terms and Privacy Policy Flow settings.
     */
    public TermsAndPrivacyPolicyFlowSettings getTermsAndPrivacyPolicyFlowSettings()
    {
        ConsentFlowSettings settings = (ConsentFlowSettings) backingConsentFlowSettings;
        settings.setConsentFlowType( ConsentFlowSettings.ConsentFlowType.UNIFIED );

        return backingConsentFlowSettings;
    }

    /**
     * Enable devices to receive test ads, by passing in the advertising identifier (GAID or App Set ID) of each test device.
     * Refer to AppLovin logs for the GAID or App Set ID of your current device.
     */
    public void setTestDeviceAdvertisingIds(final List<String> testDeviceAdvertisingIds)
    {
        Logger.logApiCall( TAG, "setTestDeviceAdvertisingIds(testDeviceAdvertisingIds=" + testDeviceAdvertisingIds + ")" );

        // Sanitize input and make copy of the list
        if ( testDeviceAdvertisingIds != null )
        {
            final List<String> sanitized = new ArrayList<>( testDeviceAdvertisingIds.size() );

            for ( final String advertisingId : testDeviceAdvertisingIds )
            {
                if ( advertisingId != null && advertisingId.length() == 36 )
                {
                    sanitized.add( advertisingId );
                }
                else
                {
                    Logger.userError( TAG, "Unable to set test device advertising id (" + advertisingId + ") - please make sure it is in the format of xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx" );
                }
            }

            this.testDeviceAdvertisingIds = sanitized;
        }
        else
        {
            this.testDeviceAdvertisingIds = Collections.emptyList();
        }
    }

    /**
     * Get the list of advertising identifiers that will receive test ads.
     */
    public List<String> getTestDeviceAdvertisingIds()
    {
        return testDeviceAdvertisingIds;
    }

    /**
     * Set the MAX ad unit ids that will be used for this instance of the SDK. 3rd-party SDKs will be initialized with the credentials configured for these ad unit ids.
     */
    public void setInitializationAdUnitIds(final List<String> initializationAdUnitIds)
    {
        Logger.logApiCall( TAG, "setInitializationAdUnitIds(initializationAdUnitIds=" + initializationAdUnitIds + ")" );

        // Sanitize input and make copy of the list
        if ( initializationAdUnitIds != null )
        {
            final List<String> sanitized = new ArrayList<>( initializationAdUnitIds.size() );

            for ( final String initializationAdUnitId : initializationAdUnitIds )
            {
                // Check for empty string from upstream deserialization of "" into [""], ignore...
                if ( StringUtils.isValidString( initializationAdUnitId ) && initializationAdUnitId.length() > 0 )
                {
                    // Correct length
                    if ( initializationAdUnitId.length() == 16 )
                    {
                        sanitized.add( initializationAdUnitId );
                    }
                    // Incorrect length
                    else
                    {
                        Logger.userError( TAG, "Unable to set initialization ad unit id (" + initializationAdUnitId + ") - please make sure it is in the format of XXXXXXXXXXXXXXXX" );
                    }
                }
            }

            this.initializationAdUnitIds = sanitized;
        }
        else
        {
            this.initializationAdUnitIds = Collections.emptyList();
        }
    }

    /**
     * Get the list of MAX ad unit ids that will be used for this instance of the SDK.
     */
    public List<String> getInitializationAdUnitIds()
    {
        return initializationAdUnitIds;
    }

    /**
     * Toggle verbose logging of AppLovin SDK. If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     *
     * @param isVerboseLoggingEnabled True if log messages should be output.
     */
    public void setVerboseLogging(boolean isVerboseLoggingEnabled)
    {
        Logger.logApiCall( TAG, "setVerboseLogging(isVerboseLoggingEnabled=" + isVerboseLoggingEnabled + ")" );

        // If enabled from Android manifest, ignore programmatic setting.
        // This makes life easier for PubOps folks when mediation networks override this setting.
        if ( Utils.isVerboseLoggingConfigured() )
        {
            Logger.userError( TAG, "Ignoring setting of verbose logging - it is configured from Android manifest already." );

            if ( Utils.isVerboseLoggingEnabled( null ) != isVerboseLoggingEnabled )
            {
                Logger.userError( TAG, "Attempted to programmatically set verbose logging flag to value different from value configured in Android Manifest." );
            }
        }
        else
        {
            this.isVerboseLoggingEnabled = isVerboseLoggingEnabled;
        }
    }

    /**
     * Check if verbose logging is enabled for the AppLovin SDK.
     * <p/>
     * If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     */
    public boolean isVerboseLoggingEnabled()
    {
        return isVerboseLoggingEnabled;
    }

    /**
     * Whether video ads begin in a muted state or not. Defaults to {@code false}.
     */
    public boolean isMuted()
    {
        return muted;
    }

    /**
     * Set whether to begin video ads in a muted state or not.
     *
     * @param muted If ads should begin in a muted state.
     */
    public void setMuted(boolean muted)
    {
        Logger.logApiCall( TAG, "setMuted(muted=" + muted + ")" );

        this.muted = muted;
    }

    /**
     * Set whether the Creative Debugger will be displayed after flipping the device screen down twice. Defaults to {@code true}.
     */
    public void setCreativeDebuggerEnabled(boolean creativeDebuggerEnabled)
    {
        Logger.logApiCall( TAG, "setCreativeDebuggerEnabled(creativeDebuggerEnabled=" + creativeDebuggerEnabled + ")" );

        if ( this.creativeDebuggerEnabled == creativeDebuggerEnabled ) return;

        this.creativeDebuggerEnabled = creativeDebuggerEnabled;

        if ( sdk == null ) return;

        if ( creativeDebuggerEnabled )
        {
            sdk.getCreativeDebuggerService().maybePrepareCreativeDebugger();
        }
        else
        {
            sdk.getCreativeDebuggerService().maybeDisableCreativeDebugger();
        }
    }

    /**
     * Whether the Creative Debugger will be displayed after flipping the device screen down twice. Defaults to {@code true}.
     */
    public boolean isCreativeDebuggerEnabled()
    {
        return creativeDebuggerEnabled;
    }

    /**
     * Set whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    public void setExceptionHandlerEnabled(boolean exceptionHandlerEnabled)
    {
        Logger.logApiCall( TAG, "setExceptionHandlerEnabled(exceptionHandlerEnabled=" + exceptionHandlerEnabled + ")" );

        this.exceptionHandlerEnabled = exceptionHandlerEnabled;
    }

    /**
     * Whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    public boolean isExceptionHandlerEnabled()
    {
        return exceptionHandlerEnabled;
    }

    /**
     * Set whether or not the AppLovin SDK will collect the device location if available. Defaults to {@code true}.
     */
    public void setLocationCollectionEnabled(final boolean locationCollectionEnabled)
    {
        Logger.logApiCall( TAG, "setLocationCollectionEnabled(locationCollectionEnabled=" + locationCollectionEnabled + ")" );

        this.locationCollectionEnabled = locationCollectionEnabled;
    }

    /**
     * Whether or not the AppLovin SDK will collect the device location if available.  Defaults to {@code true}.
     */
    public boolean isLocationCollectionEnabled()
    {
        return locationCollectionEnabled;
    }

    /**
     * A copy of the extra parameters that are currently set.
     */
    public Map<String, String> getExtraParameters()
    {
        synchronized ( extraParametersLock )
        {
            // Note: Returning a copy of the extra parameters so that our copy may not be modified by the publisher.
            return CollectionUtils.map( extraParameters );
        }
    }

    /**
     * Set an extra parameter to pass to the AppLovin server.
     *
     * @param key   Parameter key. Must not be null.
     * @param value Parameter value. May be null.
     */
    public void setExtraParameter(final String key, @Nullable final String value)
    {
        Logger.logApiCall( TAG, "setExtraParameter(key=" + key + ", value=" + value + ")" );

        if ( TextUtils.isEmpty( key ) )
        {
            Logger.userError( TAG, "Failed to set extra parameter for null or empty key: " + key );
            return;
        }

        String sanitizedValue = ( value != null ) ? value.trim() : null;

        if ( "test_mode_network".equalsIgnoreCase( key ) )
        {
            if ( sdk != null )
            {
                if ( StringUtils.isValidString( sanitizedValue ) )
                {
                    sdk.getTestModeService().setNetworks( Arrays.asList( sanitizedValue.split( "," ) ) );
                }
                else
                {
                    sdk.getTestModeService().setNetwork( null );
                }
            }
            else
            {
                testModeNetworkToSet = sanitizedValue;
            }
        }
        else if ( AppLovinSdkExtraParameterKey.FILTER_AD_NETWORK.equals( key ) || AppLovinSdkExtraParameterKey.ENABLE_SEQUENTIAL_CACHING.equals( key ) )
        {
            // Temp hack for Unity to test - their test apps' package names start with com.unity.
            if ( !packageName.startsWith( "com.unity." ) ) return;
        }

        synchronized ( extraParametersLock )
        {
            extraParameters.put( key, sanitizedValue );
        }
    }

    /**
     * The SDK will automatically fail fullscreen ad display and invoke the {@code com.applovin.mediation.MaxAdListener#onAdDisplayFailed(...)}
     * when the "Don't Keep Activities" developer setting is enabled. This setting allows bypassing that.
     * <p>
     * This flag will only be honored in debuggable builds.
     *
     * @param shouldFailAdDisplayIfDontKeepActivitiesIsEnabled Set to @{code false} to disable the SDK automatically failing fullscreen ad display when the "Don't Keep Activities" developer setting is enabled.
     */
    public void setShouldFailAdDisplayIfDontKeepActivitiesIsEnabled(final boolean shouldFailAdDisplayIfDontKeepActivitiesIsEnabled)
    {
        Logger.logApiCall( TAG, "setShouldFailAdDisplayIfDontKeepActivitiesIsEnabled(shouldFailAdDisplayIfDontKeepActivitiesIsEnabled="
                + shouldFailAdDisplayIfDontKeepActivitiesIsEnabled + ")" );

        this.failAdDisplayIfDontKeepActivitiesIsEnabled = shouldFailAdDisplayIfDontKeepActivitiesIsEnabled;
    }

    /**
     * Whether or not the SDK will automatically fail fullscreen ad display when the "Don't Keep Activities" developer setting is enabled. Defaults to {@code true}.
     */
    public boolean shouldFailAdDisplayIfDontKeepActivitiesIsEnabled()
    {
        return failAdDisplayIfDontKeepActivitiesIsEnabled;
    }

    protected void attachAppLovinSdk(CoreSdk sdk)
    {
        this.sdk = sdk;

        if ( StringUtils.isValidString( testModeNetworkToSet ) )
        {
            sdk.getTestModeService().setNetworks( Arrays.asList( testModeNetworkToSet.split( "," ) ) );

            testModeNetworkToSet = null;
        }
    }

    @SuppressLint("DiscouragedApi")
    private void updateExtraParametersFromRawResources(final Context context)
    {
        @RawRes val settingsRawJsonResourceId = context.getResources().getIdentifier( ConsentFlowManager.KEY_APPLOVIN_SDK_SETTINGS_RAW_RESOURCE_NAME, "raw", context.getPackageName() );
        if ( settingsRawJsonResourceId == 0 ) return;

        val settingsJsonString = Utils.getRawResourceString( settingsRawJsonResourceId, context, null );
        val settingsJson = StringUtils.isValidString( settingsJsonString ) ? JsonUtils.jsonObjectFromJsonString( settingsJsonString, new JSONObject() ) : new JSONObject();
        val parameters = JsonUtils.tryToStringMap( settingsJson );

        synchronized ( extraParametersLock )
        {
            extraParameters.putAll( parameters );
        }
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release.
     * <p>
     * Use the new Unified Consent Flow instead (see {@link #getTermsAndPrivacyPolicyFlowSettings()}).
     */
    @Deprecated
    public TermsFlowSettings getTermsFlowSettings()
    {
        ConsentFlowSettings settings = (ConsentFlowSettings) backingConsentFlowSettings;
        settings.setConsentFlowType( ConsentFlowSettings.ConsentFlowType.TERMS );

        return backingConsentFlowSettings;
    }

    @Override
    @NonNull
    public String toString()
    {
        return "AppLovinSdkSettings{" +
                "isVerboseLoggingEnabled=" + isVerboseLoggingEnabled +
                ", muted=" + muted +
                ", testDeviceAdvertisingIds=" + testDeviceAdvertisingIds.toString() +
                ", initializationAdUnitIds=" + initializationAdUnitIds.toString() +
                ", creativeDebuggerEnabled=" + creativeDebuggerEnabled +
                ", exceptionHandlerEnabled=" + exceptionHandlerEnabled +
                ", locationCollectionEnabled=" + locationCollectionEnabled +
                '}';
    }
}
