package com.applovin.mediation.hybridAds;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;

import com.applovin.impl.mediation.model.MediatedFullscreenAd;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.utils.ActivityLifecycleCallbacksAdapter;
import com.applovin.impl.sdk.utils.Utils;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.listeners.MaxAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxAppOpenAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;

import lombok.AllArgsConstructor;
import lombok.val;

@AllArgsConstructor
public class MaxHybridAdService
{
    private static final String TAG = "MaxHybridAdService";

    private final CoreSdk sdk;

    /**
     * Show a fullscreen native/MREC hybrid ad for the provided ad using MAX.
     *
     * @param ad       The native/MREC hybrid ad to display as a fullscreen ad. This only supports interstitial and app open ads.
     * @param activity The Activity passed in from the adapter.
     * @param listener The adapter's listener. Traditionally this is abstracted away in each adapter, but it is consolidated for hybrid ads.
     */
    public void showFullscreenAd(final MediatedFullscreenAd ad, final Activity activity, final MaxAdapterListener listener)
    {
        Utils.assertMainThread();

        val userActivity = ( activity != null ) ? activity : sdk.getActivityLifecycleManager().getTopActivity();
        if ( ad.getNativeAd() != null )
        {
            sdk.getLogger().d( TAG, "Showing fullscreen native ad..." );

            val initializer = new MaxHybridNativeAdActivityInitializer( ad, sdk, listener );
            sdk.getActivityLifecycleManager().registerActivityCallback( initializer );

            val intent = new Intent( userActivity, MaxHybridNativeAdActivity.class );
            userActivity.startActivity( intent );
        }
        else if ( ad.getAdView() != null )
        {
            sdk.getLogger().d( TAG, "Showing fullscreen MREC ad..." );

            val initializer = new MaxHybridMrecAdActivityInitializer( ad, sdk, listener );
            sdk.getActivityLifecycleManager().registerActivityCallback( initializer );

            val intent = new Intent( userActivity, MaxHybridMRecAdActivity.class );
            userActivity.startActivity( intent );
        }
        else
        {
            if ( listener instanceof MaxInterstitialAdapterListener )
            {
                ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }
            else if ( listener instanceof MaxAppOpenAdapterListener )
            {
                ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }
            else
            {
                throw new IllegalStateException( "Failed to display hybrid ad: neither native nor adview ad" );
            }
        }
    }

    @AllArgsConstructor
    private static class MaxHybridNativeAdActivityInitializer
            extends ActivityLifecycleCallbacksAdapter
    {
        private final MediatedFullscreenAd ad;
        private final CoreSdk              sdk;
        private final MaxAdapterListener   listener;

        @Override
        public void onActivityCreated(final Activity activity, final Bundle savedInstanceState)
        {
            if ( activity instanceof MaxHybridNativeAdActivity )
            {
                ( (MaxHybridNativeAdActivity) activity ).initialize( ad.getHybridAdConfiguration(), ad.getNativeAd(), sdk, listener );
            }
        }

        @Override
        public void onActivityDestroyed(final Activity activity)
        {
            if ( activity instanceof MaxHybridNativeAdActivity && !activity.isChangingConfigurations() && ad.getAdHiddenCalled().get() )
            {
                sdk.getActivityLifecycleManager().unregisterActivityCallback( this );
            }
        }
    }

    @AllArgsConstructor
    private static class MaxHybridMrecAdActivityInitializer
            extends ActivityLifecycleCallbacksAdapter
    {
        private final MediatedFullscreenAd ad;
        private final CoreSdk              sdk;
        private final MaxAdapterListener   listener;

        @Override
        public void onActivityCreated(final Activity activity, final Bundle savedInstanceState)
        {
            if ( activity instanceof MaxHybridMRecAdActivity )
            {
                ( (MaxHybridMRecAdActivity) activity ).initialize( ad.getHybridAdConfiguration(), ad.getAdView(), sdk, listener );
            }
        }

        @Override
        public void onActivityDestroyed(final Activity activity)
        {
            if ( activity instanceof MaxHybridMRecAdActivity && !activity.isChangingConfigurations() && ad.getAdHiddenCalled().get() )
            {
                sdk.getActivityLifecycleManager().unregisterActivityCallback( this );
            }
        }
    }
}
