package com.applovin.mediation.hybridAds;

import android.app.Activity;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;

import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.setting.Setting;
import com.applovin.impl.sdk.utils.ActivityUtils;
import com.applovin.impl.sdk.utils.ViewUtils;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.listeners.MaxAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxAppOpenAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;

import androidx.annotation.Nullable;
import lombok.val;
import lombok.var;

public class MaxHybridAdActivity
        extends Activity
        implements MaxCloseButton.MaxCloseButtonListener
{
    // Parent Objects
    protected CoreSdk                  sdk;
    protected MaxCloseButton           closeButton;
    protected MaxHybridAdConfiguration configuration = new MaxHybridAdConfiguration( null ); // Default configuration for rare cases where the activity is recreated without re-calling `initialize()`

    // Controlled Fields
    private MaxAdapterListener listener;

    protected void initialize(final MaxHybridAdConfiguration configuration, final CoreSdk sdk, final MaxAdapterListener listener)
    {
        this.sdk = sdk;
        this.configuration = configuration;
        this.listener = listener;
    }

    @Override
    protected void onCreate(@Nullable final Bundle savedInstanceState)
    {
        super.onCreate( savedInstanceState );

        requestWindowFeature( Window.FEATURE_NO_TITLE );
        getWindow().setFlags( WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN );
        getWindow().addFlags( WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED );
        getWindow().addFlags( WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON );

        val rootView = (ViewGroup) findViewById( android.R.id.content );
        rootView.setBackgroundColor( configuration.getBackgroundColor() );

        // `sdk` can be null in rare cases where the activity is recreated without re-calling `initialize()`
        val shouldUseInsetsController = ( sdk != null ) ? sdk.get( Setting.FULLSCREEN_AD_SHOULD_USE_INSETS_CONTROLLER ) : true;
        ActivityUtils.hideSystemUi( shouldUseInsetsController, this );

        closeButton = new MaxCloseButton( configuration, this );
        closeButton.setListener( this );
        closeButton.setVisibility( View.INVISIBLE );

        rootView.addView( closeButton );
        ViewUtils.fadeIn( closeButton, configuration.getCloseButtonDelayMillis() );

        // TODO: Implement OnBackInvokedDispatcher
    }

    //region Close Button

    @Override
    public void onCloseButtonTapReceived(final MaxCloseButton closeButton)
    {
        if ( !isFinishing() )
        {
            finish();
        }
    }

    @Override
    protected void onDestroy()
    {
        if ( listener != null )
        {
            if ( listener instanceof MaxInterstitialAdapterListener )
            {
                ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdHidden();
            }
            else if ( listener instanceof MaxAppOpenAdapterListener )
            {
                ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdHidden();
            }
            else
            {
                throw new IllegalStateException( "Failed to fire hidden callback (" + listener + "): neither interstitial nor app open ad" );
            }
        }

        super.onDestroy();
    }

    //endregion

    protected void handleAdDisplay(final View adView, final String tag)
    {
        // This should never happen but one publisher encountered a crash
        if ( adView == null )
        {
            handleAdDisplayFailed( tag );
            return;
        }

        val rootView = (ViewGroup) findViewById( android.R.id.content );
        rootView.addView( adView );

        closeButton.bringToFront();

        if ( listener instanceof MaxInterstitialAdapterListener )
        {
            ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdDisplayed();
        }
        else if ( listener instanceof MaxAppOpenAdapterListener )
        {
            ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdDisplayed();
        }
        else
        {
            throw new IllegalStateException( "Failed to fire display callback (" + listener + "): neither interstitial nor app open ad" );
        }
    }

    private void handleAdDisplayFailed(final String tag)
    {
        var errorMessage = "Null hybrid ad view (" + tag + ")";
        val adapterError = new MaxAdapterError( MaxAdapterError.ERROR_CODE_AD_DISPLAY_FAILED, errorMessage );

        if ( listener instanceof MaxInterstitialAdapterListener )
        {
            ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdDisplayFailed( adapterError );
        }
        else if ( listener instanceof MaxAppOpenAdapterListener )
        {
            ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdDisplayFailed( adapterError );
        }
        else
        {
            throw new IllegalStateException( "Failed to fire display failed callback (" + listener + "): neither interstitial nor app open ad" );
        }

        finish();
    }
}
