package com.applovin.adview;

import android.content.Context;

import com.applovin.impl.adview.InterstitialAdDialogWrapper;
import com.applovin.impl.sdk.Logger;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdSize;
import com.applovin.sdk.AppLovinSdk;

import java.lang.ref.WeakReference;

/**
 * This class represents an interstitial ad that is rendered on top of the current activity.
 *
 * @author Basil Shikin
 */
public class AppLovinInterstitialAd
{
    private static final String TAG = "AppLovinInterstitialAd";

    private static final Object                      lastCreatedDialogLock = new Object();
    private static       InterstitialAdDialogWrapper lastCreatedDialog;

    private static WeakReference<Context> lastContextRef = new WeakReference<>( null );

    /**
     * Show a new interstitial ad. This method will display a dialog on top of current activity's view with an advertisement in it.
     *
     * @param context Parent activity or application context. Must not be null.
     */
    public static void show(final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        getLastCreatedDialog( AppLovinSdk.getInstance( context ), context ).show();
    }

    /**
     * Create a new interstitial dialog that can be shown to the user. This is primarily useful in advanced integrations as the Ad Dialog allows finer control - including manually pre-caching and rendering ads. If all you want to do is show an interstitial the default way, you're looking for
     * <code>show()</code>.
     *
     * @param sdk     An SDK instance to use.
     * @param context A non-stale reference to the current top activity.
     *
     * @return A new instance of {@link com.applovin.adview.AppLovinInterstitialAdDialog}.
     */
    public static AppLovinInterstitialAdDialog create(final AppLovinSdk sdk, final Context context)
    {
        // Check input
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        return getLastCreatedDialog( sdk, context );
    }

    private static InterstitialAdDialogWrapper getLastCreatedDialog(final AppLovinSdk sdk, final Context context)
    {
        synchronized ( lastCreatedDialogLock )
        {
            if ( lastCreatedDialog == null || lastContextRef.get() != context )
            {
                lastCreatedDialog = new InterstitialAdDialogWrapper( sdk, context );
                lastContextRef = new WeakReference<>( context );
            }
            else
            {
                Logger.userError( TAG, "An interstitial is already showing" );
            }
        }

        return lastCreatedDialog;
    }

    /**
     * @deprecated Checking whether an ad is ready for display has been deprecated and will be removed in a future SDK version. Please use {@link AppLovinInterstitialAd#show(Context)}, or create an instance of {@link AppLovinInterstitialAdDialog} and use {@link AppLovinInterstitialAdDialog#show()} or {@link AppLovinInterstitialAdDialog#showAndRender(AppLovinAd)} to display an ad.
     */
    @Deprecated
    public static boolean isAdReadyToDisplay(final Context context)
    {
        return AppLovinSdk.getInstance( context ).getAdService().hasPreloadedAd( AppLovinAdSize.INTERSTITIAL );
    }

    @Override
    public String toString()
    {
        return "AppLovinInterstitialAd{}";
    }
}
