/*
 * Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 * with the License. A copy of the License is located at
 *
 * http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package com.amazonaws.serverless.proxy.jersey;


import com.amazonaws.serverless.exceptions.InvalidRequestEventException;
import com.amazonaws.serverless.proxy.internal.RequestReader;
import com.amazonaws.serverless.proxy.internal.model.AwsProxyRequest;
import com.amazonaws.serverless.proxy.internal.model.ContainerConfig;
import com.amazonaws.services.lambda.runtime.Context;
import org.glassfish.jersey.internal.MapPropertiesDelegate;
import org.glassfish.jersey.internal.PropertiesDelegate;
import org.glassfish.jersey.server.ContainerRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.UriBuilder;

import java.io.ByteArrayInputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Base64;


/**
 * Default implementation of the <code>RequestReader</code> object. This object reads an incoming <code>AwsProxyRequest</code>
 * event and transform it into a Jersey <code>ContainerRequest</code> object. The object sets three custom properties in the
 * request's <code>PropertiesDelegate</code> object: The API Gateway request context, the Map of stage variables, and the
 * Lambda context object.
 *
 * The <code>useStageAsBasePath</code> configuration variable lets you set whether the stage name should be included in the
 * request path passed to the Jersey application handler.
 */
public class JerseyAwsProxyRequestReader extends RequestReader<AwsProxyRequest, ContainerRequest> {

    //-------------------------------------------------------------
    // Variables - Private - Static
    //-------------------------------------------------------------

    private static AwsProxyRequest currentRequest;
    private static Context currentLambdaContext;

    private Logger log = LoggerFactory.getLogger(JerseyAwsProxyRequestReader.class);


    //-------------------------------------------------------------
    // Methods - Implementation
    //-------------------------------------------------------------

    /**
     * Reads an request object generated by an AWS_PROXY integration in API Gateway and transforms it into a Jersey
     * <code>ContainerRequest</code> object.
     *
     * @param request The incoming request object
     * @param securityContext A jax-rs SecurityContext object (@see com.amazonaws.serverless.proxy.internal.SecurityContextWriter)
     * @param lambdaContext The AWS Lambda context for the request
     * @param config The container config object, this is passed in by the LambdaContainerHandler
     * @return A populated ContainerRequest object
     * @throws InvalidRequestEventException When the method fails to parse the incoming request
     */
    @Override
    public ContainerRequest readRequest(AwsProxyRequest request, SecurityContext securityContext, Context lambdaContext, ContainerConfig config)
            throws InvalidRequestEventException {
        currentRequest = request;
        currentLambdaContext = lambdaContext;

        request.setPath(stripBasePath(request.getPath(), config));

        URI basePathUri;
        URI requestPathUri;
        String basePath = "/";

        try {
            basePathUri = new URI(basePath);
        } catch (URISyntaxException e) {
            log.error("Could not read base path URI", e);
            throw new InvalidRequestEventException("Error while generating base path URI: " + basePath, e);
        }


        UriBuilder uriBuilder = UriBuilder.fromPath(request.getPath());

        if (request.getQueryStringParameters() != null) {
            for (String paramKey : request.getQueryStringParameters().keySet()) {
                uriBuilder = uriBuilder.queryParam(paramKey, request.getQueryStringParameters().get(paramKey));
            }
        }

        requestPathUri = uriBuilder.build();

        PropertiesDelegate apiGatewayProperties = new MapPropertiesDelegate();
        apiGatewayProperties.setProperty(API_GATEWAY_CONTEXT_PROPERTY, request.getRequestContext());
        apiGatewayProperties.setProperty(API_GATEWAY_STAGE_VARS_PROPERTY, request.getStageVariables());
        apiGatewayProperties.setProperty(LAMBDA_CONTEXT_PROPERTY, lambdaContext);

        ContainerRequest requestContext = new ContainerRequest(basePathUri, requestPathUri, request.getHttpMethod(), securityContext, apiGatewayProperties);

        if (request.getBody() != null) {
            if (request.isBase64Encoded()) {
                requestContext.setEntityStream(new ByteArrayInputStream(Base64.getDecoder().decode(request.getBody())));
            } else {
                requestContext.setEntityStream(new ByteArrayInputStream(request.getBody().getBytes()));
            }
        }

        if (request.getHeaders() != null) {
            for (final String headerName : request.getHeaders().keySet()) {
                requestContext.headers(headerName, request.getHeaders().get(headerName));
            }
        }

        return requestContext;
    }

    //-------------------------------------------------------------
    // Methods - Protected
    //-------------------------------------------------------------

    @Override
    protected Class<? extends AwsProxyRequest> getRequestClass() {
        return AwsProxyRequest.class;
    }


    //-------------------------------------------------------------
    // Methods - Package
    //-------------------------------------------------------------

    public static AwsProxyRequest getCurrentRequest() {
        return currentRequest;
    }


    public static Context getCurrentLambdaContext() {
        return currentLambdaContext;
    }
}
