/*
 * Copyright 2020-2025 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package com.amazonaws.services.fis;

import org.w3c.dom.*;

import java.net.*;
import java.util.*;

import javax.annotation.Generated;

import org.apache.commons.logging.*;

import com.amazonaws.*;
import com.amazonaws.annotation.SdkInternalApi;
import com.amazonaws.auth.*;

import com.amazonaws.handlers.*;
import com.amazonaws.http.*;
import com.amazonaws.internal.*;
import com.amazonaws.internal.auth.*;
import com.amazonaws.metrics.*;
import com.amazonaws.regions.*;
import com.amazonaws.transform.*;
import com.amazonaws.util.*;
import com.amazonaws.protocol.json.*;
import com.amazonaws.util.AWSRequestMetrics.Field;
import com.amazonaws.annotation.ThreadSafe;
import com.amazonaws.client.AwsSyncClientParams;
import com.amazonaws.client.builder.AdvancedConfig;

import com.amazonaws.services.fis.AWSFISClientBuilder;

import com.amazonaws.AmazonServiceException;

import com.amazonaws.services.fis.model.*;

import com.amazonaws.services.fis.model.transform.*;

/**
 * Client for accessing FIS. All service calls made using this client are blocking, and will not return until the
 * service call completes.
 * <p>
 * <p>
 * Fault Injection Service is a managed service that enables you to perform fault injection experiments on your Amazon
 * Web Services workloads. For more information, see the <a
 * href="https://docs.aws.amazon.com/fis/latest/userguide/">Fault Injection Service User Guide</a>.
 * </p>
 */
@ThreadSafe
@Generated("com.amazonaws:aws-java-sdk-code-generator")
public class AWSFISClient extends AmazonWebServiceClient implements AWSFIS {

    /** Provider for AWS credentials. */
    private final AWSCredentialsProvider awsCredentialsProvider;

    private static final Log log = LogFactory.getLog(AWSFIS.class);

    /** Default signing name for the service. */
    private static final String DEFAULT_SIGNING_NAME = "fis";

    /** Client configuration factory providing ClientConfigurations tailored to this client */
    protected static final ClientConfigurationFactory configFactory = new ClientConfigurationFactory();

    private final AdvancedConfig advancedConfig;

    private static final com.amazonaws.protocol.json.SdkJsonProtocolFactory protocolFactory = new com.amazonaws.protocol.json.SdkJsonProtocolFactory(
            new JsonClientMetadata()
                    .withProtocolVersion("1.1")
                    .withSupportsCbor(false)
                    .withSupportsIon(false)
                    .withContentTypeOverride("application/json")
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ServiceQuotaExceededException").withExceptionUnmarshaller(
                                    com.amazonaws.services.fis.model.transform.ServiceQuotaExceededExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ConflictException").withExceptionUnmarshaller(
                                    com.amazonaws.services.fis.model.transform.ConflictExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withExceptionUnmarshaller(
                                    com.amazonaws.services.fis.model.transform.ResourceNotFoundExceptionUnmarshaller.getInstance()))
                    .addErrorMetadata(
                            new JsonErrorShapeMetadata().withErrorCode("ValidationException").withExceptionUnmarshaller(
                                    com.amazonaws.services.fis.model.transform.ValidationExceptionUnmarshaller.getInstance()))
                    .withBaseServiceExceptionClass(com.amazonaws.services.fis.model.AWSFISException.class));

    public static AWSFISClientBuilder builder() {
        return AWSFISClientBuilder.standard();
    }

    /**
     * Constructs a new client to invoke service methods on FIS using the specified parameters.
     *
     * <p>
     * All service calls made using this new client object are blocking, and will not return until the service call
     * completes.
     *
     * @param clientParams
     *        Object providing client parameters.
     */
    AWSFISClient(AwsSyncClientParams clientParams) {
        this(clientParams, false);
    }

    /**
     * Constructs a new client to invoke service methods on FIS using the specified parameters.
     *
     * <p>
     * All service calls made using this new client object are blocking, and will not return until the service call
     * completes.
     *
     * @param clientParams
     *        Object providing client parameters.
     */
    AWSFISClient(AwsSyncClientParams clientParams, boolean endpointDiscoveryEnabled) {
        super(clientParams);
        this.awsCredentialsProvider = clientParams.getCredentialsProvider();
        this.advancedConfig = clientParams.getAdvancedConfig();
        init();
    }

    private void init() {
        setServiceNameIntern(DEFAULT_SIGNING_NAME);
        setEndpointPrefix(ENDPOINT_PREFIX);
        // calling this.setEndPoint(...) will also modify the signer accordingly
        setEndpoint("fis.us-east-1.amazonaws.com");
        HandlerChainFactory chainFactory = new HandlerChainFactory();
        requestHandler2s.addAll(chainFactory.newRequestHandlerChain("/com/amazonaws/services/fis/request.handlers"));
        requestHandler2s.addAll(chainFactory.newRequestHandler2Chain("/com/amazonaws/services/fis/request.handler2s"));
        requestHandler2s.addAll(chainFactory.getGlobalHandlers());
    }

    /**
     * <p>
     * Creates an experiment template.
     * </p>
     * <p>
     * An experiment template includes the following components:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Targets</b>: A target can be a specific resource in your Amazon Web Services environment, or one or more
     * resources that match criteria that you specify, for example, resources that have specific tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Actions</b>: The actions to carry out on the target. You can specify multiple actions, the duration of each
     * action, and when to start each action during an experiment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Stop conditions</b>: If a stop condition is triggered while an experiment is running, the experiment is
     * automatically stopped. You can define a stop condition as a CloudWatch alarm.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fis/latest/userguide/experiment-templates.html">experiment templates</a> in the
     * <i>Fault Injection Service User Guide</i>.
     * </p>
     * 
     * @param createExperimentTemplateRequest
     * @return Result of the CreateExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ConflictException
     *         The request could not be processed because of a conflict.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @sample AWSFIS.CreateExperimentTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/CreateExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateExperimentTemplateResult createExperimentTemplate(CreateExperimentTemplateRequest request) {
        request = beforeClientExecution(request);
        return executeCreateExperimentTemplate(request);
    }

    @SdkInternalApi
    final CreateExperimentTemplateResult executeCreateExperimentTemplate(CreateExperimentTemplateRequest createExperimentTemplateRequest) {

        ExecutionContext executionContext = createExecutionContext(createExperimentTemplateRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CreateExperimentTemplateRequest> request = null;
        Response<CreateExperimentTemplateResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CreateExperimentTemplateRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(createExperimentTemplateRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CreateExperimentTemplate");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CreateExperimentTemplateResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateExperimentTemplateResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Creates a target account configuration for the experiment template. A target account configuration is required
     * when <code>accountTargeting</code> of <code>experimentOptions</code> is set to <code>multi-account</code>. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/fis/latest/userguide/experiment-options.html">experiment options</a> in the
     * <i>Fault Injection Service User Guide</i>.
     * </p>
     * 
     * @param createTargetAccountConfigurationRequest
     * @return Result of the CreateTargetAccountConfiguration operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ConflictException
     *         The request could not be processed because of a conflict.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @sample AWSFIS.CreateTargetAccountConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/CreateTargetAccountConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateTargetAccountConfigurationResult createTargetAccountConfiguration(CreateTargetAccountConfigurationRequest request) {
        request = beforeClientExecution(request);
        return executeCreateTargetAccountConfiguration(request);
    }

    @SdkInternalApi
    final CreateTargetAccountConfigurationResult executeCreateTargetAccountConfiguration(
            CreateTargetAccountConfigurationRequest createTargetAccountConfigurationRequest) {

        ExecutionContext executionContext = createExecutionContext(createTargetAccountConfigurationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<CreateTargetAccountConfigurationRequest> request = null;
        Response<CreateTargetAccountConfigurationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new CreateTargetAccountConfigurationRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(createTargetAccountConfigurationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "CreateTargetAccountConfiguration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<CreateTargetAccountConfigurationResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateTargetAccountConfigurationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Deletes the specified experiment template.
     * </p>
     * 
     * @param deleteExperimentTemplateRequest
     * @return Result of the DeleteExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.DeleteExperimentTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/DeleteExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteExperimentTemplateResult deleteExperimentTemplate(DeleteExperimentTemplateRequest request) {
        request = beforeClientExecution(request);
        return executeDeleteExperimentTemplate(request);
    }

    @SdkInternalApi
    final DeleteExperimentTemplateResult executeDeleteExperimentTemplate(DeleteExperimentTemplateRequest deleteExperimentTemplateRequest) {

        ExecutionContext executionContext = createExecutionContext(deleteExperimentTemplateRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<DeleteExperimentTemplateRequest> request = null;
        Response<DeleteExperimentTemplateResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new DeleteExperimentTemplateRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(deleteExperimentTemplateRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "DeleteExperimentTemplate");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<DeleteExperimentTemplateResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteExperimentTemplateResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Deletes the specified target account configuration of the experiment template.
     * </p>
     * 
     * @param deleteTargetAccountConfigurationRequest
     * @return Result of the DeleteTargetAccountConfiguration operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.DeleteTargetAccountConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/DeleteTargetAccountConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTargetAccountConfigurationResult deleteTargetAccountConfiguration(DeleteTargetAccountConfigurationRequest request) {
        request = beforeClientExecution(request);
        return executeDeleteTargetAccountConfiguration(request);
    }

    @SdkInternalApi
    final DeleteTargetAccountConfigurationResult executeDeleteTargetAccountConfiguration(
            DeleteTargetAccountConfigurationRequest deleteTargetAccountConfigurationRequest) {

        ExecutionContext executionContext = createExecutionContext(deleteTargetAccountConfigurationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<DeleteTargetAccountConfigurationRequest> request = null;
        Response<DeleteTargetAccountConfigurationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new DeleteTargetAccountConfigurationRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(deleteTargetAccountConfigurationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "DeleteTargetAccountConfiguration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<DeleteTargetAccountConfigurationResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteTargetAccountConfigurationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified FIS action.
     * </p>
     * 
     * @param getActionRequest
     * @return Result of the GetAction operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.GetAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetAction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetActionResult getAction(GetActionRequest request) {
        request = beforeClientExecution(request);
        return executeGetAction(request);
    }

    @SdkInternalApi
    final GetActionResult executeGetAction(GetActionRequest getActionRequest) {

        ExecutionContext executionContext = createExecutionContext(getActionRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetActionRequest> request = null;
        Response<GetActionResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetActionRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getActionRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetAction");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetActionResult>> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                    .withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetActionResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified experiment.
     * </p>
     * 
     * @param getExperimentRequest
     * @return Result of the GetExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.GetExperiment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExperimentResult getExperiment(GetExperimentRequest request) {
        request = beforeClientExecution(request);
        return executeGetExperiment(request);
    }

    @SdkInternalApi
    final GetExperimentResult executeGetExperiment(GetExperimentRequest getExperimentRequest) {

        ExecutionContext executionContext = createExecutionContext(getExperimentRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetExperimentRequest> request = null;
        Response<GetExperimentResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetExperimentRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getExperimentRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetExperiment");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetExperimentResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new GetExperimentResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified target account configuration of the experiment.
     * </p>
     * 
     * @param getExperimentTargetAccountConfigurationRequest
     * @return Result of the GetExperimentTargetAccountConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.GetExperimentTargetAccountConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetExperimentTargetAccountConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetExperimentTargetAccountConfigurationResult getExperimentTargetAccountConfiguration(GetExperimentTargetAccountConfigurationRequest request) {
        request = beforeClientExecution(request);
        return executeGetExperimentTargetAccountConfiguration(request);
    }

    @SdkInternalApi
    final GetExperimentTargetAccountConfigurationResult executeGetExperimentTargetAccountConfiguration(
            GetExperimentTargetAccountConfigurationRequest getExperimentTargetAccountConfigurationRequest) {

        ExecutionContext executionContext = createExecutionContext(getExperimentTargetAccountConfigurationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetExperimentTargetAccountConfigurationRequest> request = null;
        Response<GetExperimentTargetAccountConfigurationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetExperimentTargetAccountConfigurationRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(getExperimentTargetAccountConfigurationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetExperimentTargetAccountConfiguration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetExperimentTargetAccountConfigurationResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new GetExperimentTargetAccountConfigurationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified experiment template.
     * </p>
     * 
     * @param getExperimentTemplateRequest
     * @return Result of the GetExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.GetExperimentTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetExperimentTemplate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExperimentTemplateResult getExperimentTemplate(GetExperimentTemplateRequest request) {
        request = beforeClientExecution(request);
        return executeGetExperimentTemplate(request);
    }

    @SdkInternalApi
    final GetExperimentTemplateResult executeGetExperimentTemplate(GetExperimentTemplateRequest getExperimentTemplateRequest) {

        ExecutionContext executionContext = createExecutionContext(getExperimentTemplateRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetExperimentTemplateRequest> request = null;
        Response<GetExperimentTemplateResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetExperimentTemplateRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getExperimentTemplateRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetExperimentTemplate");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetExperimentTemplateResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new GetExperimentTemplateResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified target account configuration of the experiment template.
     * </p>
     * 
     * @param getTargetAccountConfigurationRequest
     * @return Result of the GetTargetAccountConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.GetTargetAccountConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetTargetAccountConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTargetAccountConfigurationResult getTargetAccountConfiguration(GetTargetAccountConfigurationRequest request) {
        request = beforeClientExecution(request);
        return executeGetTargetAccountConfiguration(request);
    }

    @SdkInternalApi
    final GetTargetAccountConfigurationResult executeGetTargetAccountConfiguration(GetTargetAccountConfigurationRequest getTargetAccountConfigurationRequest) {

        ExecutionContext executionContext = createExecutionContext(getTargetAccountConfigurationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetTargetAccountConfigurationRequest> request = null;
        Response<GetTargetAccountConfigurationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetTargetAccountConfigurationRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(getTargetAccountConfigurationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetTargetAccountConfiguration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetTargetAccountConfigurationResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new GetTargetAccountConfigurationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Gets information about the specified resource type.
     * </p>
     * 
     * @param getTargetResourceTypeRequest
     * @return Result of the GetTargetResourceType operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.GetTargetResourceType
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetTargetResourceType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTargetResourceTypeResult getTargetResourceType(GetTargetResourceTypeRequest request) {
        request = beforeClientExecution(request);
        return executeGetTargetResourceType(request);
    }

    @SdkInternalApi
    final GetTargetResourceTypeResult executeGetTargetResourceType(GetTargetResourceTypeRequest getTargetResourceTypeRequest) {

        ExecutionContext executionContext = createExecutionContext(getTargetResourceTypeRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<GetTargetResourceTypeRequest> request = null;
        Response<GetTargetResourceTypeResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new GetTargetResourceTypeRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(getTargetResourceTypeRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "GetTargetResourceType");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<GetTargetResourceTypeResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new GetTargetResourceTypeResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the available FIS actions.
     * </p>
     * 
     * @param listActionsRequest
     * @return Result of the ListActions operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.ListActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListActions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListActionsResult listActions(ListActionsRequest request) {
        request = beforeClientExecution(request);
        return executeListActions(request);
    }

    @SdkInternalApi
    final ListActionsResult executeListActions(ListActionsRequest listActionsRequest) {

        ExecutionContext executionContext = createExecutionContext(listActionsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListActionsRequest> request = null;
        Response<ListActionsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListActionsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listActionsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListActions");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListActionsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListActionsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the resolved targets information of the specified experiment.
     * </p>
     * 
     * @param listExperimentResolvedTargetsRequest
     * @return Result of the ListExperimentResolvedTargets operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.ListExperimentResolvedTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperimentResolvedTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExperimentResolvedTargetsResult listExperimentResolvedTargets(ListExperimentResolvedTargetsRequest request) {
        request = beforeClientExecution(request);
        return executeListExperimentResolvedTargets(request);
    }

    @SdkInternalApi
    final ListExperimentResolvedTargetsResult executeListExperimentResolvedTargets(ListExperimentResolvedTargetsRequest listExperimentResolvedTargetsRequest) {

        ExecutionContext executionContext = createExecutionContext(listExperimentResolvedTargetsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListExperimentResolvedTargetsRequest> request = null;
        Response<ListExperimentResolvedTargetsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListExperimentResolvedTargetsRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listExperimentResolvedTargetsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListExperimentResolvedTargets");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListExperimentResolvedTargetsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListExperimentResolvedTargetsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the target account configurations of the specified experiment.
     * </p>
     * 
     * @param listExperimentTargetAccountConfigurationsRequest
     * @return Result of the ListExperimentTargetAccountConfigurations operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.ListExperimentTargetAccountConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperimentTargetAccountConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExperimentTargetAccountConfigurationsResult listExperimentTargetAccountConfigurations(ListExperimentTargetAccountConfigurationsRequest request) {
        request = beforeClientExecution(request);
        return executeListExperimentTargetAccountConfigurations(request);
    }

    @SdkInternalApi
    final ListExperimentTargetAccountConfigurationsResult executeListExperimentTargetAccountConfigurations(
            ListExperimentTargetAccountConfigurationsRequest listExperimentTargetAccountConfigurationsRequest) {

        ExecutionContext executionContext = createExecutionContext(listExperimentTargetAccountConfigurationsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListExperimentTargetAccountConfigurationsRequest> request = null;
        Response<ListExperimentTargetAccountConfigurationsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListExperimentTargetAccountConfigurationsRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listExperimentTargetAccountConfigurationsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListExperimentTargetAccountConfigurations");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListExperimentTargetAccountConfigurationsResult>> responseHandler = protocolFactory
                    .createResponseHandler(new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new ListExperimentTargetAccountConfigurationsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists your experiment templates.
     * </p>
     * 
     * @param listExperimentTemplatesRequest
     * @return Result of the ListExperimentTemplates operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.ListExperimentTemplates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperimentTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListExperimentTemplatesResult listExperimentTemplates(ListExperimentTemplatesRequest request) {
        request = beforeClientExecution(request);
        return executeListExperimentTemplates(request);
    }

    @SdkInternalApi
    final ListExperimentTemplatesResult executeListExperimentTemplates(ListExperimentTemplatesRequest listExperimentTemplatesRequest) {

        ExecutionContext executionContext = createExecutionContext(listExperimentTemplatesRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListExperimentTemplatesRequest> request = null;
        Response<ListExperimentTemplatesResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListExperimentTemplatesRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listExperimentTemplatesRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListExperimentTemplates");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListExperimentTemplatesResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListExperimentTemplatesResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists your experiments.
     * </p>
     * 
     * @param listExperimentsRequest
     * @return Result of the ListExperiments operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.ListExperiments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExperimentsResult listExperiments(ListExperimentsRequest request) {
        request = beforeClientExecution(request);
        return executeListExperiments(request);
    }

    @SdkInternalApi
    final ListExperimentsResult executeListExperiments(ListExperimentsRequest listExperimentsRequest) {

        ExecutionContext executionContext = createExecutionContext(listExperimentsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListExperimentsRequest> request = null;
        Response<ListExperimentsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListExperimentsRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listExperimentsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListExperiments");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListExperimentsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListExperimentsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     * 
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @sample AWSFIS.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResult listTagsForResource(ListTagsForResourceRequest request) {
        request = beforeClientExecution(request);
        return executeListTagsForResource(request);
    }

    @SdkInternalApi
    final ListTagsForResourceResult executeListTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(listTagsForResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListTagsForResourceRequest> request = null;
        Response<ListTagsForResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListTagsForResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(listTagsForResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListTagsForResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListTagsForResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new ListTagsForResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the target account configurations of the specified experiment template.
     * </p>
     * 
     * @param listTargetAccountConfigurationsRequest
     * @return Result of the ListTargetAccountConfigurations operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.ListTargetAccountConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListTargetAccountConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTargetAccountConfigurationsResult listTargetAccountConfigurations(ListTargetAccountConfigurationsRequest request) {
        request = beforeClientExecution(request);
        return executeListTargetAccountConfigurations(request);
    }

    @SdkInternalApi
    final ListTargetAccountConfigurationsResult executeListTargetAccountConfigurations(
            ListTargetAccountConfigurationsRequest listTargetAccountConfigurationsRequest) {

        ExecutionContext executionContext = createExecutionContext(listTargetAccountConfigurationsRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListTargetAccountConfigurationsRequest> request = null;
        Response<ListTargetAccountConfigurationsResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListTargetAccountConfigurationsRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listTargetAccountConfigurationsRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListTargetAccountConfigurations");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListTargetAccountConfigurationsResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListTargetAccountConfigurationsResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Lists the target resource types.
     * </p>
     * 
     * @param listTargetResourceTypesRequest
     * @return Result of the ListTargetResourceTypes operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @sample AWSFIS.ListTargetResourceTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListTargetResourceTypes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTargetResourceTypesResult listTargetResourceTypes(ListTargetResourceTypesRequest request) {
        request = beforeClientExecution(request);
        return executeListTargetResourceTypes(request);
    }

    @SdkInternalApi
    final ListTargetResourceTypesResult executeListTargetResourceTypes(ListTargetResourceTypesRequest listTargetResourceTypesRequest) {

        ExecutionContext executionContext = createExecutionContext(listTargetResourceTypesRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<ListTargetResourceTypesRequest> request = null;
        Response<ListTargetResourceTypesResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new ListTargetResourceTypesRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(listTargetResourceTypesRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "ListTargetResourceTypes");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<ListTargetResourceTypesResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListTargetResourceTypesResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Starts running an experiment from the specified experiment template.
     * </p>
     * 
     * @param startExperimentRequest
     * @return Result of the StartExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ConflictException
     *         The request could not be processed because of a conflict.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @sample AWSFIS.StartExperiment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/StartExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartExperimentResult startExperiment(StartExperimentRequest request) {
        request = beforeClientExecution(request);
        return executeStartExperiment(request);
    }

    @SdkInternalApi
    final StartExperimentResult executeStartExperiment(StartExperimentRequest startExperimentRequest) {

        ExecutionContext executionContext = createExecutionContext(startExperimentRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<StartExperimentRequest> request = null;
        Response<StartExperimentResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new StartExperimentRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(startExperimentRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "StartExperiment");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<StartExperimentResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new StartExperimentResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Stops the specified experiment.
     * </p>
     * 
     * @param stopExperimentRequest
     * @return Result of the StopExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.StopExperiment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/StopExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopExperimentResult stopExperiment(StopExperimentRequest request) {
        request = beforeClientExecution(request);
        return executeStopExperiment(request);
    }

    @SdkInternalApi
    final StopExperimentResult executeStopExperiment(StopExperimentRequest stopExperimentRequest) {

        ExecutionContext executionContext = createExecutionContext(stopExperimentRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<StopExperimentRequest> request = null;
        Response<StopExperimentResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new StopExperimentRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(stopExperimentRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "StopExperiment");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<StopExperimentResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new StopExperimentResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Applies the specified tags to the specified resource.
     * </p>
     * 
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @sample AWSFIS.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResult tagResource(TagResourceRequest request) {
        request = beforeClientExecution(request);
        return executeTagResource(request);
    }

    @SdkInternalApi
    final TagResourceResult executeTagResource(TagResourceRequest tagResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(tagResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<TagResourceRequest> request = null;
        Response<TagResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new TagResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(tagResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "TagResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<TagResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new TagResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     * 
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @sample AWSFIS.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResult untagResource(UntagResourceRequest request) {
        request = beforeClientExecution(request);
        return executeUntagResource(request);
    }

    @SdkInternalApi
    final UntagResourceResult executeUntagResource(UntagResourceRequest untagResourceRequest) {

        ExecutionContext executionContext = createExecutionContext(untagResourceRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UntagResourceRequest> request = null;
        Response<UntagResourceResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UntagResourceRequestProtocolMarshaller(protocolFactory).marshall(super.beforeMarshalling(untagResourceRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UntagResource");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UntagResourceResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false), new UntagResourceResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Updates the specified experiment template.
     * </p>
     * 
     * @param updateExperimentTemplateRequest
     * @return Result of the UpdateExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @sample AWSFIS.UpdateExperimentTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/UpdateExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateExperimentTemplateResult updateExperimentTemplate(UpdateExperimentTemplateRequest request) {
        request = beforeClientExecution(request);
        return executeUpdateExperimentTemplate(request);
    }

    @SdkInternalApi
    final UpdateExperimentTemplateResult executeUpdateExperimentTemplate(UpdateExperimentTemplateRequest updateExperimentTemplateRequest) {

        ExecutionContext executionContext = createExecutionContext(updateExperimentTemplateRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UpdateExperimentTemplateRequest> request = null;
        Response<UpdateExperimentTemplateResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UpdateExperimentTemplateRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(updateExperimentTemplateRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UpdateExperimentTemplate");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UpdateExperimentTemplateResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateExperimentTemplateResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * <p>
     * Updates the target account configuration for the specified experiment template.
     * </p>
     * 
     * @param updateTargetAccountConfigurationRequest
     * @return Result of the UpdateTargetAccountConfiguration operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @sample AWSFIS.UpdateTargetAccountConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/UpdateTargetAccountConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTargetAccountConfigurationResult updateTargetAccountConfiguration(UpdateTargetAccountConfigurationRequest request) {
        request = beforeClientExecution(request);
        return executeUpdateTargetAccountConfiguration(request);
    }

    @SdkInternalApi
    final UpdateTargetAccountConfigurationResult executeUpdateTargetAccountConfiguration(
            UpdateTargetAccountConfigurationRequest updateTargetAccountConfigurationRequest) {

        ExecutionContext executionContext = createExecutionContext(updateTargetAccountConfigurationRequest);
        AWSRequestMetrics awsRequestMetrics = executionContext.getAwsRequestMetrics();
        awsRequestMetrics.startEvent(Field.ClientExecuteTime);
        Request<UpdateTargetAccountConfigurationRequest> request = null;
        Response<UpdateTargetAccountConfigurationResult> response = null;

        try {
            awsRequestMetrics.startEvent(Field.RequestMarshallTime);
            try {
                request = new UpdateTargetAccountConfigurationRequestProtocolMarshaller(protocolFactory).marshall(super
                        .beforeMarshalling(updateTargetAccountConfigurationRequest));
                // Binds the request metrics to the current request.
                request.setAWSRequestMetrics(awsRequestMetrics);
                request.addHandlerContext(HandlerContextKey.CLIENT_ENDPOINT, endpoint);
                request.addHandlerContext(HandlerContextKey.ENDPOINT_OVERRIDDEN, isEndpointOverridden());
                request.addHandlerContext(HandlerContextKey.SIGNING_REGION, getSigningRegion());
                request.addHandlerContext(HandlerContextKey.SERVICE_ID, "fis");
                request.addHandlerContext(HandlerContextKey.OPERATION_NAME, "UpdateTargetAccountConfiguration");
                request.addHandlerContext(HandlerContextKey.ADVANCED_CONFIG, advancedConfig);

            } finally {
                awsRequestMetrics.endEvent(Field.RequestMarshallTime);
            }

            HttpResponseHandler<AmazonWebServiceResponse<UpdateTargetAccountConfigurationResult>> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateTargetAccountConfigurationResultJsonUnmarshaller());
            response = invoke(request, responseHandler, executionContext);

            return response.getAwsResponse();

        } finally {

            endClientExecution(awsRequestMetrics, request, response);
        }
    }

    /**
     * Returns additional metadata for a previously executed successful, request, typically used for debugging issues
     * where a service isn't acting as expected. This data isn't considered part of the result data returned by an
     * operation, so it's available through this separate, diagnostic interface.
     * <p>
     * Response metadata is only cached for a limited period of time, so if you need to access this extra diagnostic
     * information for an executed request, you should use this method to retrieve it as soon as possible after
     * executing the request.
     *
     * @param request
     *        The originally executed request
     *
     * @return The response metadata for the specified request, or null if none is available.
     */
    public ResponseMetadata getCachedResponseMetadata(AmazonWebServiceRequest request) {
        return client.getResponseMetadataForRequest(request);
    }

    /**
     * Normal invoke with authentication. Credentials are required and may be overriden at the request level.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> invoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext) {

        return invoke(request, responseHandler, executionContext, null, null);
    }

    /**
     * Normal invoke with authentication. Credentials are required and may be overriden at the request level.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> invoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext, URI cachedEndpoint, URI uriFromEndpointTrait) {

        executionContext.setCredentialsProvider(CredentialUtils.getCredentialsProvider(request.getOriginalRequest(), awsCredentialsProvider));

        return doInvoke(request, responseHandler, executionContext, cachedEndpoint, uriFromEndpointTrait);
    }

    /**
     * Invoke with no authentication. Credentials are not required and any credentials set on the client or request will
     * be ignored for this operation.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> anonymousInvoke(Request<Y> request,
            HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler, ExecutionContext executionContext) {

        return doInvoke(request, responseHandler, executionContext, null, null);
    }

    /**
     * Invoke the request using the http client. Assumes credentials (or lack thereof) have been configured in the
     * ExecutionContext beforehand.
     **/
    private <X, Y extends AmazonWebServiceRequest> Response<X> doInvoke(Request<Y> request, HttpResponseHandler<AmazonWebServiceResponse<X>> responseHandler,
            ExecutionContext executionContext, URI discoveredEndpoint, URI uriFromEndpointTrait) {

        if (discoveredEndpoint != null) {
            request.setEndpoint(discoveredEndpoint);
            request.getOriginalRequest().getRequestClientOptions().appendUserAgent("endpoint-discovery");
        } else if (uriFromEndpointTrait != null) {
            request.setEndpoint(uriFromEndpointTrait);
        } else {
            request.setEndpoint(endpoint);
        }

        request.setTimeOffset(timeOffset);

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());

        return client.execute(request, responseHandler, errorResponseHandler, executionContext);
    }

    @com.amazonaws.annotation.SdkInternalApi
    static com.amazonaws.protocol.json.SdkJsonProtocolFactory getProtocolFactory() {
        return protocolFactory;
    }

    @Override
    public void shutdown() {
        super.shutdown();
    }

}
