/*
 * Decompiled with CFR 0.152.
 */
package com.aliyun.core.http.policy;

import com.aliyun.core.http.HttpHeader;
import com.aliyun.core.http.HttpHeaders;
import com.aliyun.core.http.HttpRequest;
import com.aliyun.core.http.HttpResponse;
import com.aliyun.core.http.policy.HttpLogDetailLevel;
import com.aliyun.core.http.policy.HttpLogOptions;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.logging.LogLevel;
import com.aliyun.core.utils.StringUtils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.Collections;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class HttpLoggingPolicy {
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintBody;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
            this.prettyPrintBody = false;
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames().stream().map(headerName -> headerName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames().stream().map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
        }
    }

    private void logRequest(ClientLogger logger, HttpRequest request, Optional<Object> optionalRetryCount) {
        long contentLength;
        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return;
        }
        StringBuilder requestLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            requestLogMessage.append("--> ").append((Object)request.getHttpMethod()).append(" ").append(System.lineSeparator());
            optionalRetryCount.ifPresent(o -> requestLogMessage.append("Try count: ").append(o).append(System.lineSeparator()));
        }
        this.addHeadersToLogMessage(logger, request.getHeaders(), requestLogMessage);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            this.logAndReturn(logger, requestLogMessage, null);
            return;
        }
        if (request.getBody() == null) {
            requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            this.logAndReturn(logger, requestLogMessage, null);
            return;
        }
        String contentType = request.getHeaders().getValue("Content-Type");
        if (this.shouldBodyBeLogged(contentType, contentLength = this.getContentLength(logger, request.getHeaders()))) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(HttpLoggingPolicy.convertStreamToString(outputStream, logger)).append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            logger.info(requestLogMessage.toString());
        } else {
            requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            this.logAndReturn(logger, requestLogMessage, null);
        }
    }

    private HttpResponse logResponse(ClientLogger logger, HttpResponse response, long startNs) throws IOException {
        long contentLength;
        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return response;
        }
        long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
        String contentLengthString = response.getHeaderValue("Content-Length");
        String bodySize = StringUtils.isEmpty(contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
        StringBuilder responseLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(" (").append(tookMs).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
        }
        this.addHeadersToLogMessage(logger, response.getHeaders(), responseLogMessage);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            responseLogMessage.append("<-- END HTTP");
            return this.logAndReturn(logger, responseLogMessage, response);
        }
        String contentTypeHeader = response.getHeaderValue("Content-Type");
        if (this.shouldBodyBeLogged(contentTypeHeader, contentLength = this.getContentLength(logger, response.getHeaders()))) {
            HttpResponse bufferedResponse = response.buffer();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            HttpLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, bufferedResponse.getBody());
            responseLogMessage.append("Response body:").append(System.lineSeparator()).append(HttpLoggingPolicy.convertStreamToString(outputStream, logger)).append(System.lineSeparator()).append("<-- END HTTP");
            logger.info(responseLogMessage.toString());
            return bufferedResponse;
        }
        responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
        return this.logAndReturn(logger, responseLogMessage, response);
    }

    private <T> T logAndReturn(ClientLogger logger, StringBuilder logMessageBuilder, T data) {
        logger.info(logMessageBuilder.toString());
        return data;
    }

    private String getAllowedQueryString(String queryString) {
        String[] queryParams;
        if (StringUtils.isEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (this.allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private void addHeadersToLogMessage(ClientLogger logger, HttpHeaders headers, StringBuilder sb) {
        if (!this.httpLogDetailLevel.shouldLogHeaders() || !logger.canLogAtLevel(LogLevel.VERBOSE)) {
            return;
        }
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (this.allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (StringUtils.isEmpty(contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }

    private boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }

    private static String convertStreamToString(ByteArrayOutputStream stream, ClientLogger logger) {
        try {
            return stream.toString("UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            throw logger.logExceptionAsError(new RuntimeException(ex));
        }
    }

    private static CompletableFuture<ByteBuffer> writeBufferToBodyStream(WritableByteChannel channel, ByteBuffer byteBuffer) throws IOException {
        channel.write(byteBuffer.duplicate());
        return CompletableFuture.completedFuture(byteBuffer);
    }
}

