//______________________________________________________________________________________
//
//  WifiSensor.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.sensors;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;


import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.NetworkInfo;
import android.net.wifi.ScanResult;
import android.net.wifi.SupplicantState;
import android.net.wifi.WifiConfiguration;
import android.net.wifi.WifiManager;
import android.net.wifi.WifiManager.WifiLock;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;

import com.polestar.helpers.ConnectivityHelper;
import com.polestar.models.ISensorListener;
import com.polestar.helpers.Log;
import com.polestar.models.SensorNao;
import com.polestar.models.WifiMeasurement;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.TSENSORPOWERMODE;
import com.polestar.naosdk.api.TSENSORTYPE;

/**
 * Manages the WiFi interface (scans, etc.) for Android
 *    
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 *   <td>Author</td>
 *   <td>Modification date</td>
 *   <td>Tracking Number</td>
 *   <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>4 mars 2010</td>
 *   <td>trunk</td>
 *   <td>Creation of this class</td>
 * </tr>
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>15 mars 2010</td>
 *   <td>trunk</td>
 *   <td>can now work launched by a service</td>
 * </tr>
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>15 mars 2010</td>
 *   <td>trunk</td>
 *   <td>moved from WifiSensor to NaoNativeBridge</td>
 * </tr>
 * </table>    
 * @author sterrenoir
 */
public class WifiSensor extends SensorNao {
	private static final String WIFI_SENSOR_THREAD_NAME = "WS_Thread"; // name of the thread WIFI
	/**
	 * Minimum time allowed for scan results to be considered as complete after a scan request<br/>
	 * unit : [ms] <br/>
	 * typical value 560ms is 40ms (time to scan 1 channel according to standards (reference TBD)) * 14 channels
	 * 900ms allows more time for native threads to run and catch the messages
	*/
	private static final long MINIMUM_TIME_FOR_SCAN_REQUEST = 1000;
	
	/** Maximum time allowed between 2 sendings of wifi info to the mSensorObserver.
	 *  This is also monitored in the native and could be pre-empted by a reset.
	 *  Unit = milliseconds.
	 *  IMPORTANT: Must remain coherent with timers defined in sensors_manager.h!
	 *  i.e. >reset time && <log gap time
	 */
	private static final long MAX_TIME_WITHOUT_WIFI_UPDATE_MS = 6000;
	/**
	 * Minimum time span in milliseconds between a scan request and a reset request
	 * Very slow-scanning devices will be forced to reset before they have a chance to answer,
	 * but it does not matter because this type of devices all fail to recover from disconnection.
	 */
	private static final long MIN_TIME_BEFORE_RESETS = 5000;

	/**
	 * RSSI filter threshold   
	 */
	private static final int RSSI_ROAMING_THRESHOLD = -70;
	/**
	 * minimum time between two successive roaming 
	 * time in msecond
	 */
	private static final int MINIMUM_TIME_BETWEEN_ROAMING = 10000;
	/** 
	 * the time required for OS when roaming
	 * time in msecond
	 */
	private static final int REQUIRED_TIME_FOR_ROAMING = 2500;
	
	// variables
	/** Intent filter that allows notification of new Wi-Fi scan results */
	private IntentFilter mScanIntentFilter = null;
	/** Intent filter that allows notification of supplicant state changes */
	private IntentFilter mStateChangeIntentFilter = null;
	/** Intent filter that allows notification of overall network connection state changes */
	private IntentFilter mNetStateIntentFilter = null;
	/** Reference to the Android OS Wi-Fi manager, to trigger any action */
	private WifiManager mWifiMng = null;
	/** Tells whether Wi-Fi was enabled when the user started the application */
	private boolean mWasWifiEnabledAtAppStart = false;
	private long mScanStartTime, mScanStopTime;
	private long mScanResultsSentTime = 0;
	private WifiMeasurement mCurrentWifiMeas = null;
	/** remanent list of wifi scan results, to send to NAO Logger in some configurations */
	private WifiMeasurement mLastWifiMeas = null;
	private WifiLock mWifiLock = null;
	private WifiBroadcastReceiver mWifiScannerEventReceiver = null;
	private WiFiStateChangeBroadcastReceiver mWiFiStateChangeEventReceiver = null;
	private NetworkStateChangeBroadcastReceiver mNetStateChangeEventReceiver = null;
	private boolean mWifiCollectorReadyToSend = false;


	
	// messages to perform thread actions in the Handler
	private static final int MSG_READY_TO_SEND = 1;
	private static final int MSG_START = 2;
	private static final int MSG_STOP = 3;
	private static final int MSG_RESET = 4;
	
	// status of the Reset actions
	/** Tells if a reset request is either waiting to be processed or being processed */
	private boolean mResetIsPendingOrInProgress = false;
	
	private boolean mfirstTurnWifiOn = false;
	/** if this is registered, scan results will be sent this way */
	private ISensorListener mTargetInterface = null;

	/** list of WifiConfiguration objects top remember what connections were active when the sensor was launched, 
	 * in case the sensor disables them (for internal logging especially)
	 */
	private ArrayList<WifiConfiguration> mWifiConnectionProfiles = null;
	/** Tells whether the sensor shall take every possible step to maximize Wifi info acquisition,
	 *  (potentially) at the cost of the user's  = nullexperience of other applications in parallel to the Nao service. 
	 */
	private boolean mShallOptimizeForLogging = false;
	/** Tells is the wifi connections have been disabled yet */
	private boolean mWifiConnectionDisabled = false;

	/** roaming DBG********************************************************/
	// TODO - enlever le commentaire "DBG" ou supprimer cet attribut!
	private String lastNetworkBSSID = null;
	/********************************************************************/
	/** roaming detection flag*/
	boolean isRoamingBeenDetected = false;
	
	/** time since the last roaming action*/
	private long mTimeSinceLastRomaing = -1;
	private long mLastTimeRomaing = 0;
	
	private static boolean bWifiBroadcastReceiverRegistered = false;
	private static boolean bWiFiStateChangeBroadcastReceiverRegistered = false;
	private static boolean bNetworkStateChangeBroadcastReceiverRegistered = false;
	
	private static WifiCollectorHandler mWifiCollectorHandler = null;

	// create handler to process incoming measurements
	/** nested static class to define the Handler and its actions */
	private static class WifiCollectorHandler extends Handler {
		private WeakReference<WifiSensor> wifiSensorRef;
		
		public WifiCollectorHandler(WifiSensor ws) {
			wifiSensorRef = new WeakReference<WifiSensor>(ws);
		}
		
		public void handleMessage(Message msg) {
			WifiSensor ws = wifiSensorRef.get();
			if (msg == null || ws == null)
				return;
			switch (msg.what) {
				case MSG_READY_TO_SEND:
					// set flag to allow sending measurements to client
					ws.mWifiCollectorReadyToSend = true;
					// if wifi data has been collected during the current scan, send it immediately.
					if ( (ws.mCurrentWifiMeas!= null) && (ws.mCurrentWifiMeas.getNumberOfAccessPoints()>0)) {
						ws.processScanResults(false);
					}
					break;
				case MSG_START:
					ws.startSensorImpl();
					break;
				case MSG_STOP:
					ws.stopSensorImpl();
					break;
				case MSG_RESET:
					ws.resetImpl();
					break;
				default:
					Log.alwaysError(this.getClass().getName(), "Unexpected message!");
					break;
			}
		}
	} // end of WifiCollectorHandler definition
		
	/** a broadcast receiver class to process incoming Wi-Fi scan results and supplicant state changes */
	private class WifiBroadcastReceiver extends BroadcastReceiver {
		public void onReceive(Context c, Intent i) {			
			//Log.restricted(this.getClass().getName(),"onReceive getScanResults().size()="+((mWifiMng==null)?-1:((mWifiMng.getScanResults()==null)?-1:mWifiMng.getScanResults().size())));
			processScanResults(true);
        }
        
        public void register() {
            if ((mCtxtWrap != null) && (mScanIntentFilter != null)  && (!bWifiBroadcastReceiverRegistered)) {
            	mCtxtWrap.registerReceiver(this, mScanIntentFilter);
            	bWifiBroadcastReceiverRegistered = true;
            }
        }
        public void unregister() {
            if (bWifiBroadcastReceiverRegistered && mCtxtWrap != null) {
            	try {
            		mCtxtWrap.unregisterReceiver(this);
            		bWifiBroadcastReceiverRegistered = false;
            	} catch (IllegalArgumentException e) {
        			Log.alwaysError(this.getClass().getName(), e.toString()) ;
        		}
            }
        }
	} // end of WifiBroadcastReceiver definition
	
	
	/**
	 * A broadcast receiver to closely monitor changes in WiFi connection state
	 */
	private class WiFiStateChangeBroadcastReceiver extends BroadcastReceiver {
		private SupplicantState previousState = null;
		
		public void onReceive(Context c, Intent i) {
	    	//--- log additional data to detect log gaps ---
			SupplicantState currentState = i.getParcelableExtra(WifiManager.EXTRA_NEW_STATE);
			//Log.restricted(this.getClass().getName(),"WiFi state --> "+currentState.name());
			//--- end log additional data ---
			// launch a reset if we've just been disconnected
			if ((previousState != null) && 
					(previousState == SupplicantState.COMPLETED) && 
					(currentState == SupplicantState.DISCONNECTED)) {
				reset();
			}
			previousState = currentState;
        }
        public void register() {
        	if ((mCtxtWrap != null) && (mStateChangeIntentFilter != null)  && (!bWiFiStateChangeBroadcastReceiverRegistered)) {
            	mCtxtWrap.registerReceiver(this, mStateChangeIntentFilter);
            	bWiFiStateChangeBroadcastReceiverRegistered = true;
        	}
        }
        public void unregister() {
        	if ((bWiFiStateChangeBroadcastReceiverRegistered) && (mCtxtWrap != null)) {
        		try {
	        		mCtxtWrap.unregisterReceiver(this);
	        	    bWiFiStateChangeBroadcastReceiverRegistered = false;
        		} catch (IllegalArgumentException e) {
        			Log.alwaysError(this.getClass().getName(), e.toString()) ;
        		}
        	}
        }
	} // end of WiFiStateChangeBroadcastReceiver definition
	
	
	/**
	 * A broadcast receiver to closely monitor changes in WiFi connection state
	 */
	private class NetworkStateChangeBroadcastReceiver extends BroadcastReceiver {
		public void onReceive(Context c, Intent i) {
			NetworkInfo netInfo = (NetworkInfo)i.getParcelableExtra(WifiManager.EXTRA_NETWORK_INFO);
	    	//--- log additional data to detect log gaps ---
			Log.restricted(this.getClass().getName(),"Net state --> "+netInfo.getState().name());
			//--- end log additional data ---
        }
        public void register() {
        	if ((mCtxtWrap != null) && (mNetStateIntentFilter != null)  && (!bNetworkStateChangeBroadcastReceiverRegistered)) {
        	    mCtxtWrap.registerReceiver(this, mNetStateIntentFilter);
        	    bNetworkStateChangeBroadcastReceiverRegistered = true;
        	}
        }
        public void unregister() {
        	if (bNetworkStateChangeBroadcastReceiverRegistered) {
        		try {
        			mCtxtWrap.unregisterReceiver(this);
        			bNetworkStateChangeBroadcastReceiverRegistered = false;
        		} catch (IllegalArgumentException e) {
        			Log.alwaysError(this.getClass().getName(), e.toString()) ;
        		}
        	}
        }
	} // end of NetworkStateChangeBroadcastReceiver definition
	
	
	/**
	 * 
	 * WifiSensor constructor : creates an intent filter, get a handle to the wifi services, etc.
	 * This constructor is called when launched by an activity.
	 * @param caller: reference to calling activity or service
	 * @param _shallOptimizeStandaloneOperations: if true, allows the sensor to take every possible action
	 * to maximize Wifi data acquisition, at the potential cost of other Wifi activity from other applications.
	 */
	public WifiSensor(ContextWrapper caller, ISensorObserver observer,boolean _shallOptimizeStandaloneOperations) {

		setSensorObserver(observer);

		mThread = new Thread(this);
		mThread.setName(WIFI_SENSOR_THREAD_NAME);
		mThread.start();

		mStatus = STATUS_OFF;
		// check input reference
		if (caller == null) {
    		mStatus = STATUS_ERROR;
		}
		// copy the reference to the calling activity or service
		mCtxtWrap = caller;

		// get a reference to the Wifi manager service
		mWifiMng = (WifiManager) mCtxtWrap.getSystemService(Context.WIFI_SERVICE);
		
		// control whether all wifi profiles will be disconnected or not
		mShallOptimizeForLogging = _shallOptimizeStandaloneOperations;
		
		//init last time roaming
		mLastTimeRomaing = System.currentTimeMillis();
	} // end of WifiSensor
	
	
	/**
	 * Thread entry point, initializes the object.
	 */
	public void run() {
		Looper.prepare();
		myLooper = Looper.myLooper();

		// Initialize the handler to perform actions in this thread
		mWifiCollectorHandler = new WifiCollectorHandler(this);
		
		this.mOrder = ORDER_TURN_OFF ;
		
		init();
		
		Looper.loop();
	}
	
	/**
	 * Destructor : cleanly stops the service and put all references to null
	 */
	protected void finalize() {
		// clean-up all references	
		mTargetInterface = null;
		mCtxtWrap = null;
		mWifiMng = null;
	}
	
	/**
	 * Initializes the Wifi Sensor at the start of the thread<br>
	 * Has no effect if status is not OFF
	 */
	private synchronized void init() {		
		if (mStatus != STATUS_OFF) {
			// this should not happen
			return;
		}
        
        if (this.mShallOptimizeForLogging) {
			// TelephonyManager listener allows monitoring of cellular data connections, to prevent long gaps:
			PhoneStateListener myPhoneStateListener = new PhoneStateListener() {
				/** Oldest time from which only "disconnected" states have been reported */
				public long oldestContinuousDisconnectedStateTime = 0;
				public int lastState = TelephonyManager.DATA_DISCONNECTED;
				/** Whether a connection has occurred since the last report of a data disconnection to the sensor */
				public boolean haveSeenConnectionAtLeastOnceSinceLastDisconnect = false;
				public void onDataConnectionStateChanged(int state) {
			    	//--- log additional data to detect log gaps ---
	    			String stateStr;
					switch (state) {
						case TelephonyManager.DATA_DISCONNECTED:
							stateStr = "DATA_DISCONNECTED";
							break;
						case TelephonyManager.DATA_CONNECTING:
							stateStr = "DATA_CONNECTING";
							break;
						case TelephonyManager.DATA_CONNECTED:
							stateStr = "DATA_CONNECTED";
							break;
						case TelephonyManager.DATA_SUSPENDED:
							stateStr = "DATA_SUSPENDED";
							break;
						default:
							stateStr = "UNKNOWN";
							break;
					}
					Log.restricted(this.getClass().getName(),"Telephony Data state --> "+stateStr);
					//--- end log additional data ---
					
					// Now try and survey the connectivit else {
					// There is a high probability that staying for long in "disconnected" creates log gaps
					// but it is not sure and requires to have been connected before.
					if (state == TelephonyManager.DATA_DISCONNECTED) {
						if (lastState != TelephonyManager.DATA_DISCONNECTED)
							// Remember current time as "start of disconnected session"
							oldestContinuousDisconnectedStateTime = System.currentTimeMillis();
						else if (  (System.currentTimeMillis() - oldestContinuousDisconnectedStateTime > MAX_TIME_WITHOUT_WIFI_UPDATE_MS)
								&& (haveSeenConnectionAtLeastOnceSinceLastDisconnect)  ) {
							// It *may* be time to reset the sensor, since data disconnected usually means no Wifi
							// scan info from the WifiManager. However this is not a sure thing, so let the sensor decide.
							dataConnectivitySuggestsReset();
							haveSeenConnectionAtLeastOnceSinceLastDisconnect = false;
						}
					} else {
						// Any other state than disconnected can mean a connection has been established
						// Especially DATA_CONNECTING has been seen without a CONNECTED, but data was established
						// enough to create a measurement gap.
						if (lastState == TelephonyManager.DATA_DISCONNECTED) {
							haveSeenConnectionAtLeastOnceSinceLastDisconnect = true;
						}
					}
					lastState = state;
				}
			};
			TelephonyManager mTelManager = (TelephonyManager)mCtxtWrap.getSystemService(Context.TELEPHONY_SERVICE);
			mTelManager.listen(myPhoneStateListener, PhoneStateListener.LISTEN_DATA_CONNECTION_STATE);
        } // else do not monitor this
        
        
        //get wifi state when starting.
        mWasWifiEnabledAtAppStart = (this.mWifiMng != null && this.mWifiMng.isWifiEnabled())? true:false;
        
        //if(NaoLocationService.checkKey() >= 3 && this.mWifiMng != null && this.mWifiMng.isWifiEnabled()== false) {
		if(this.mWifiMng != null && this.mWifiMng.isWifiEnabled()== false) {

			// turn ON the Wifi device
			if (this.turnWifiOn()) {
				mStatus = STATUS_READY;
			}
			else {
				mStatus = STATUS_ERROR;
			}

		}
		
	} // end of init()
	
	
	/** Function called when there is indication in the phone's (data) connectivity states that Wifi probably has 
	 *  not been allowed to be updated for a long time. Though this probably has no effect, a reset is attempted in 
	 *  that case.
	 */
	protected void dataConnectivitySuggestsReset() {
		// Verify how long has actually elapsed since the last info was sent:
		if (System.currentTimeMillis() - this.mScanResultsSentTime > MAX_TIME_WITHOUT_WIFI_UPDATE_MS) {
			// This means same code as reset with a Wifi turned off on top.
	    	try {
	    		if (mWifiMng.isWifiEnabled()) {
		    		mWifiMng.setWifiEnabled(false);
	    			// Give a little time for the Wi-Fi to be really off:
    				Thread.sleep(500);
		    	}
	    	} catch(SecurityException se) {
	    		Log.alwaysError(this.getClass().getName(),"SecurityException : " + se.toString());
			} catch (InterruptedException e) {
				// Thread.sleep was interrupted. Just keep going.
	    	} catch(Exception e) {
	    		Log.alwaysError(this.getClass().getName(),"Exception : " + e.toString());
			}
	    	
	    	// log this event
	    	Log.restricted(this.getClass().getName(),"dataConnectivitySuggestsReset!");

	    	// launch the reset
	    	reset();
		}
	}


	@Override
	public void start() {
		Log.alwaysWarn(this.getClass().getName(),"Starting Wifi sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_ON, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	@Override
	public boolean isRunning() {
		return (mStatus == STATUS_RUNNING);
	}

	@Override
	public void stop() {
		Log.alwaysWarn(this.getClass().getName(),"Stoping Wifi sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_OFF, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	/**
	 * Stops the sensor if running, clears all members, reinitializes all and restart if it was previously running.
	 * This can be called from any thread.
	 */
	@Override
	public void reset() {
		// if no reset is is progress, and the last scan request is older than 10s, then
		// ask the Wifi Sensor thread to do a reset
		if (this.mOrder==ORDER_TURN_ON && !mResetIsPendingOrInProgress) {
			mResetIsPendingOrInProgress = true;
			long elapsedTimeSinceLatestScanRequest = System.currentTimeMillis() - mScanStartTime;
			if (elapsedTimeSinceLatestScanRequest >= MIN_TIME_BEFORE_RESETS) {
				// last scan request was issued a long time ago
				Log.restricted(this.getClass().getName(),"MSG_RESET message posted");
				mWifiCollectorHandler.sendEmptyMessage(MSG_RESET);
			} else {
				// last scan request is pretty recent
				long delayMillis = MIN_TIME_BEFORE_RESETS-elapsedTimeSinceLatestScanRequest;
				Log.restricted(this.getClass().getName(),"MSG_RESET message posted with a "+delayMillis+"ms delay");
				mWifiCollectorHandler.sendEmptyMessageDelayed(MSG_RESET, delayMillis);				
			}			
		}

	}

	@Override
	public void setPowerMode(TSENSORPOWERMODE mode) {
		//Not implemented
	}


	/**
	 * Actual implementation of the reset mechanism. Shall run in the WifiSensor Thread.
	 */
	private void resetImpl() {
		
		if(this.mWifiMng != null && this.mWifiMng.isWifiEnabled()== false) {
			return;
		}
		
		// --- log this event in the special log file
		Log.restricted(this.getClass().getName(),"reset lvl1");
		// --- end of special log		
		int initialStatus = mStatus;
		
		//do not reset if wifi scanner is not running
		if (initialStatus != STATUS_RUNNING) {
			return;
		}
		
		// disconnect the sensor
		switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_SERVICE);
		
		// if the wifi is not connected right now, turn off the chipset and on again
		if (mWifiMng.getConnectionInfo()!=null && mWifiMng.getConnectionInfo().getSupplicantState()!=SupplicantState.COMPLETED) {
			Log.restricted(this.getClass().getName(),"reset lvl2");
			turnWifiOff();
			turnWifiOn();
		}
		
		if (initialStatus == STATUS_RUNNING) {
			switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_SERVICE);
		}
		
		mResetIsPendingOrInProgress = false;		
	}

	/**
	 * Initializes the sensor, starts the scanning events. Can be called from any thread.<br>
	 * Does <i>not</i> start the thread 
	 * <li>Locks the wifi
	 * <li>optimizes logging (disconnects...) if required
	 * <li>registers event receivers
	 * <li>start the scanning loop by launching the first scan command
	 * @return true if the method can be called in the right context
	 */
	@Override
	public boolean startSensor() {
		boolean returnValue = true;
		// If called from another thread: post execution to this thread
		if (mWifiCollectorHandler != null) {
			mWifiCollectorHandler.sendEmptyMessage(MSG_START);
		} else {
			// Error: Handler not initialized
			returnValue = false;
		}

		return returnValue;
	}
	
	/**
	 * Initializes the sensor, starts the scanning events. Shall ONLY be called from the WS thread.<br>
	 * Does <i>not</i> start the thread 
	 */
	private void startSensorImpl() {
		//get wifi state when starting.
        mWasWifiEnabledAtAppStart = (this.mWifiMng != null && this.mWifiMng.isWifiEnabled())? true:false;
		
		if(this.mWifiMng != null && this.mWifiMng.isWifiEnabled() == true) {
			mStatus = STATUS_READY;
		}
				
		if (mStatus==STATUS_READY) {

	        // get a reference to the Wifi Lock object, will lock wifi radio for scan only
	        mWifiLock = mWifiMng.createWifiLock(WifiManager.WIFI_MODE_FULL ,"com.polestar.Nao");
	        mWifiLock.setReferenceCounted(false);
	        
	        // create a Wifi Measurement object
	        mCurrentWifiMeas = new WifiMeasurement();
	        
			// disconnect Wi-Fi networks if required (no effect if already disabled or not required)
			setOptimizeForLogging(mShallOptimizeForLogging);
			
    		//Keep wifi radio awake
    		if ((mWifiLock != null) && (!mWifiLock.isHeld())) {
    			mWifiLock.acquire();
    		}
    		
	        // intent filter to get scan results
	        mScanIntentFilter = new IntentFilter();
	        mScanIntentFilter.addAction(WifiManager.SCAN_RESULTS_AVAILABLE_ACTION);
	        // intent filters to monitor Wi-Fi and Network supplicant state changes
	        mStateChangeIntentFilter = new IntentFilter();
	        mStateChangeIntentFilter.addAction(WifiManager.SUPPLICANT_STATE_CHANGED_ACTION);
	        mNetStateIntentFilter = new IntentFilter();
	        mNetStateIntentFilter.addAction(WifiManager.NETWORK_STATE_CHANGED_ACTION);
	        
	        // associated receivers
	        mWifiScannerEventReceiver = new WifiBroadcastReceiver();
	        mWiFiStateChangeEventReceiver = new WiFiStateChangeBroadcastReceiver();
 	        mNetStateChangeEventReceiver = new NetworkStateChangeBroadcastReceiver();
 	        
	        // register the asynchronous receivers
			mWifiScannerEventReceiver.register();
			mWiFiStateChangeEventReceiver.register();
			mNetStateChangeEventReceiver.register();
			
			boolean hasLaunchedOk = launchScan();
			for (int i=0; (i<9) && (!hasLaunchedOk); i++) {
				// Wait a little bit, in case this is a slow Wi-Fi chip
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					// Just keep going earlier than planned
				}
				hasLaunchedOk = launchScan();
			}
			
			//if (hasLaunchedOk) {
				mStatus = STATUS_RUNNING;
				this.mOrder = ORDER_TURN_ON;
			//}
		} else if (mStatus==STATUS_RUNNING) {
			// remove any other START messages that might be waiting at this point
			mWifiCollectorHandler.removeMessages(MSG_START);
			// No need to log - cannot start Wifi scanner (already running).
		} else {
			Log.alwaysWarn(this.getClass().getName(),"cannot start Wifi scanner (not ready)");
		}
	}

	/**
	 * Stops the Wifi sensor. Can be called from any thread.<br>
	 * <li>Stop the scanning loop
	 * <li>Unregisters all asynchronous receivers
	 * <li>Releases the Wi-Fi lock
	 */
	@Override
	public void stopSensor() {
		// If called from another thread: post execution to this thread
		if (mWifiCollectorHandler != null) {
			mWifiCollectorHandler.sendEmptyMessage(MSG_STOP);
		}

		// wait a bit until the deed is done
		try {
			for (int k=0;k<10 && mStatus==STATUS_RUNNING;k++) {
					Thread.sleep(20);
			}
		} catch (InterruptedException e) {
			// void
		}
	}

	/**
	 * Stops the Wifi sensor. Should ONLY be called from the WS thread.<br>
	 * <li>Stop the scanning loop
	 * <li>Unregisters all asynchronous receivers
	 * <li>Releases the Wi-Fi lock
	 */
	private void stopSensorImpl() {
		if (mWifiCollectorHandler != null) {
	    	// remove remaining messages from the queue if any
			mWifiCollectorHandler.removeMessages(MSG_READY_TO_SEND);
			mWifiCollectorHandler.removeMessages(MSG_RESET);
			mWifiCollectorHandler.removeMessages(MSG_START);
			mWifiCollectorHandler.removeMessages(MSG_STOP);
			// reset control flag
			mWifiCollectorReadyToSend = false;
		}
		
		
		if (mStatus==STATUS_RUNNING) {

			// unregister all receivers
			if (mWifiScannerEventReceiver != null)
				mWifiScannerEventReceiver.unregister();
			if (mWiFiStateChangeEventReceiver != null)
				mWiFiStateChangeEventReceiver.unregister();
			if (mNetStateChangeEventReceiver != null)
				mNetStateChangeEventReceiver.unregister();
					
			

			if ((mWifiLock != null) && (mWifiLock.isHeld())) {
				mWifiLock.release();
			}


			// reconnect wifi if it had been disconnected
			setOptimizeForLogging(false);
			
			mCurrentWifiMeas = null;		
			mWifiScannerEventReceiver = null;
			mWiFiStateChangeEventReceiver = null;
			mNetStateChangeEventReceiver = null;		
			mScanIntentFilter = null;
			mStateChangeIntentFilter = null;
			mNetStateIntentFilter = null;		
			mWifiLock = null;	

			// update status
			mStatus = STATUS_READY;
			this.mOrder = ORDER_TURN_OFF;
		} else {
			Log.alwaysWarn(this.getClass().getName(),"cannot stop Wifi scanner (not running)");
		}
	}
	
	
	/**
	 * set sensors  State At App Start
	 */
	public boolean setStateAsBefore() {
		
		// Also return the Wi-Fi chip to the state it was before Nao was started:
		if (mWasWifiEnabledAtAppStart == false && mWifiMng!=null && mWifiMng.getWifiState() !=  WifiManager.WIFI_STATE_DISABLED) {
			// Turn Wi-Fi off:
			if (false == mWifiMng.setWifiEnabled(false)) {
				Log.alwaysWarn(this.getClass().getName(), "Could not turn Wi-Fi back off");
			}
		} // else don't do anything since Wi-Fi is still on
		
		return true;
	}	
	
	/**
	 * 
	 * Register an interface for output wifi measurements
	 * @param listener : reference to an object implementing a ISensorListener interface
	 */
	public void registerOutputInterface(ISensorListener listener) {
		mTargetInterface = listener;
	}
	
	/**
	 * return the latest list of wifi scan results
	 */
	public WifiMeasurement getLastWifiMeas() { 
		return this.mLastWifiMeas;
	}
	
	/**
	 * Process scan result depending on the currScanResultent status and control flag<br>
	 * This method is called when a new wifi measurement arrives.<br>
	 * <b>case 1:</b><br>
	 *   collector timer is not over yet, in this case just append the new data to the current measurement<br>
	 *   call comes from WifiBroadcastReceiver.onReceive()<br>
	 * <b>case 2:</b><br>
	 *   collector timer is over, just send the data and start a new scan<br>
	 *   call comes from Handler.handleMessage()<br>	 *   
	 * @param newDataAvailable : true if there is new wifi data to collect, false else
	 */
	public synchronized void processScanResults(boolean newDataAvailable) {
    	if (mStatus==STATUS_RUNNING) { 
    		// anyway, collect the new data if available
    		if (newDataAvailable) {
    	    	mScanStopTime = System.currentTimeMillis();
    	    	// Collect latest scan results and append them to the measurement    
    	    	if(mCurrentWifiMeas!= null) {
    	    		mCurrentWifiMeas.add(mWifiMng.getScanResults(),mScanStartTime,mScanStopTime);
    	    	}
    	    	
    	    	
    	    	// cancel pending resets if scan results keep coming
    	    	if ((mResetIsPendingOrInProgress) &&  (mCurrentWifiMeas!= null) &&( mCurrentWifiMeas.getNumberOfAccessPoints()>0)) {
    	    		if(mWifiCollectorHandler!= null) {
    	    			mWifiCollectorHandler.removeMessages(MSG_RESET);
    	    		}
    	    		mResetIsPendingOrInProgress = false;
    	    		Log.restricted(this.getClass().getName(),"Removing pending MSG_RESET messages from the queue");
    	    	}
    	    }
    		
    		// send if it is the right time
    		if (mWifiCollectorReadyToSend) {
    			
    			//update time since last roaming action
    			mTimeSinceLastRomaing = System.currentTimeMillis() - mLastTimeRomaing;
    			
    			mLastWifiMeas = mCurrentWifiMeas;
    			
    	    	if (mTargetInterface != null) {
	   	    		mTargetInterface.onNewMeasurement(mCurrentWifiMeas);
    	    	}

				if(mSensorObserver != null) {
					mSensorObserver.notifyOfNewData(TSENSORTYPE.WIFI, mCurrentWifiMeas.toByteArray());
				}
    	    	
    	    	mScanResultsSentTime = System.currentTimeMillis();
    			
	        	// reset control flag
	        	mWifiCollectorReadyToSend = false;
	        	// clear internal APs buffer:
	        	mCurrentWifiMeas.clear();
	        	
	        	//check if roaming must be detected
	        	//Add try to catch NullPointerException - crash from getConnectionInfo
	        	try {
	    	    	if(isRoamingDetected() == true) {
			        	try {
		    	    			isRoamingBeenDetected = true;
		    	    			switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_SERVICE);
								Thread.sleep(REQUIRED_TIME_FOR_ROAMING);
								switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_SERVICE);
								mLastTimeRomaing = System.currentTimeMillis();
								
			    		} catch (InterruptedException e) {
								Log.alwaysError(this.getClass().getName(), e.toString()) ;
			    		}
	    	    	}
		    	    	
	    	    	if(lastNetworkBSSID == null) {
	    	    		lastNetworkBSSID = new String();
	    	    	}
    	    	
	    			if(lastNetworkBSSID!=null 
	    					&& mWifiMng != null 
	    					&& mWifiMng.getConnectionInfo()!=null 
	    					&& mWifiMng.getConnectionInfo().getSupplicantState()!=null
	    					&& (mWifiMng.getConnectionInfo().getSupplicantState().equals(SupplicantState.ASSOCIATED) 
							|| mWifiMng.getConnectionInfo().getSupplicantState().equals(SupplicantState.COMPLETED))
							&& mWifiMng.getConnectionInfo().getBSSID()!=null
	    					&& lastNetworkBSSID.equalsIgnoreCase(mWifiMng.getConnectionInfo().getBSSID()) == false) {
	    				Log.alwaysWarn(this.getClass().getName(), lastNetworkBSSID + " - [ " +mWifiMng.getConnectionInfo().getBSSID() + " ]");
	
	    				lastNetworkBSSID = mWifiMng.getConnectionInfo().getBSSID();
	    			} 
    	    	} catch (NullPointerException e) {
					Log.alwaysError(this.getClass().getName(), e.toString()) ;
    	    	}
	        	
    			// next scan
    			if(isRoamingBeenDetected == false) {
	        	launchScan();
			}
			//update roaming detection flag
			isRoamingBeenDetected = false;
    			
		}
    	}
	} // end of processScanResults()
	

	/**
	 * Detect when roaming need to be done
	 * @return true if OK, false else
	 */
	private boolean isRoamingDetected() {
 	
			boolean isRoamingNeedToBeDone = false;
			if(mWifiMng == null) {
				return false;
			}
			
			//get wifi connection state
			if(mWifiMng.getConnectionInfo() == null) {
				return false;
			}
			SupplicantState currentWifiState = mWifiMng.getConnectionInfo().getSupplicantState();
			
			if(currentWifiState == null) {
				return false;
			}
			
    		if(currentWifiState.equals(SupplicantState.ASSOCIATED)  && currentWifiState.equals(SupplicantState.COMPLETED)){ 
				return false;
    		}
			
    		// RSSI filter
    		if(mWifiMng.getConnectionInfo().getRssi() > RSSI_ROAMING_THRESHOLD) {
    			return false;
    		}
    	
    		//check if ap with the same SSID and better rssi in the ap scan list
    		List<ScanResult> wifiScanList =  mWifiMng.getScanResults();
    		if (wifiScanList != null) {
    			for (int k = 0;k<wifiScanList.size();k++) {
    				if(wifiScanList.get(k).SSID.equals(mWifiMng.getConnectionInfo().getSSID()) == true
    						&& wifiScanList.get(k).BSSID.equals(mWifiMng.getConnectionInfo().getBSSID()) == false
    						&& wifiScanList.get(k).level > mWifiMng.getConnectionInfo().getRssi()) {
    					isRoamingNeedToBeDone = true;
    					break;
    				}
    			}
    		}
    		
    		//time since last roaming filter
    		if((mTimeSinceLastRomaing != -1) && mTimeSinceLastRomaing < MINIMUM_TIME_BETWEEN_ROAMING) {
    			return false;
    		}
    		
    	//return true when roaming is need to be done
		return isRoamingNeedToBeDone;
	}

	/**
	 * Initialize the Wifi Sensor, starts the Wifi if needed.
	 * @return true if OK, false else
	 */
	private boolean turnWifiOn() {
				
		boolean return_value = true;
		// check ContextWrapper
    	if (mCtxtWrap == null) {
    		Log.alwaysError(this.getClass().getName(),"Error turnWifiOn : ContextWrapper is null!");
    		return false;
    	}
    	
    	synchronized(mWifiMng) {
			return ConnectivityHelper.turnWifiON(mCtxtWrap);
    	}

	}
	
	
	/**
	 * Turns off the wifi chipset, and waits until it is really off
	 * @return true if success
	 */
	private boolean turnWifiOff() {
		synchronized(mWifiMng) {
	    	return ConnectivityHelper.turnWifiOFF(mCtxtWrap);
		}
	}
	
	
	private synchronized boolean launchScan() {	
		if (mCurrentWifiMeas != null) {
			mCurrentWifiMeas.clear();
		}
		mScanStartTime = System.currentTimeMillis();
		boolean result = false;
		if (mWifiMng != null) {
			result = mWifiMng.startScan();
			if (!result) {
				Log.restricted(this.getClass().getName(),"Error in launchScan(): mWifiMng.startScan() failed!");
			}
		}
		if (mWifiCollectorHandler == null) {
			Log.restricted(this.getClass().getName(),"Error in launchScan(): mWifiCollectorHandler is null!");
			result = false;
		}
		if (result) {
			//wait 
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
			}
			Message tmp = Message.obtain(mWifiCollectorHandler, MSG_READY_TO_SEND);
			if (null == tmp)
				result = false;
			else
				result = mWifiCollectorHandler.sendMessageDelayed(tmp, MINIMUM_TIME_FOR_SCAN_REQUEST);
		}
		
		if (!result) {
			Log.alwaysError(this.getClass().getName(),"Error: cannot start a new wifi scan!");
		}
		return result;
	} // end of launchScan()

	@Override
	public boolean hasFix() {
		// Wifi sensors do not compute location fixes
		return false;
	}

	@Override
	public boolean isOn() {
		return (mStatus == STATUS_RUNNING);
	}
	
	/**
     * get if sensor is active
     */
	public boolean isActive() {
		// check current wifi state 
    	if (mWifiMng == null) {
    		return false;
    	}
		return mWifiMng.isWifiEnabled(); 
	}
	
	/**
     * get if sensor is HW compatible
     */
	public boolean isHardwareCompatible() {

		return (mCtxtWrap.getSystemService(Context.WIFI_SERVICE) != null); 
	}
	
	/**
	 * @return the current Wi-Fi state as a readable string
	 */
	@SuppressWarnings("unused")
	private String getWifiStateAsString() {
		switch(mWifiMng.getWifiState()) {
		case   WifiManager.WIFI_STATE_DISABLED:
			return "DISABLED";
		case   WifiManager.WIFI_STATE_DISABLING:
			return "DISABLING";
		case   WifiManager.WIFI_STATE_ENABLED:
			return "ENABLED";
		case   WifiManager.WIFI_STATE_ENABLING:
			return "ENABLING";
		case   WifiManager.WIFI_STATE_UNKNOWN:
		default:
			return "UNKNOWN";
		}
	}
	
	/**
	 * Switches the connection disable and log optimization for standalone operation feature<br> 
	 * Disconnects/reconnects the Wi-Fi profiles if needed
	 * @param shallDisable new value for the feature 
	 */
	public synchronized void setOptimizeForLogging(boolean shallDisable) {		
		// Change command value
		mShallOptimizeForLogging = shallDisable;

		// disable or enable right now if sensor has been initialized
		if ((mStatus==STATUS_READY) || (mStatus==STATUS_RUNNING)) {
			if (mShallOptimizeForLogging) {
	    		disableWifiConnections();				
			} else {
				enableWifiConnections();				
			}
		}
		// if sensor is not ready, it will be done at the StartSensor call
		
    	//--- log additional data to detect log gaps ---
		Log.restricted(this.getClass().getName(),"ws_optim="+mShallOptimizeForLogging+" disabled="+mWifiConnectionDisabled);
		//--- end log additional data ---
	}

	/**
	 * Disable all existing configurations for Wi-Fi associations:
	 */
	private void disableWifiConnections() {
		if ((mWifiMng==null) || (mWifiConnectionDisabled))
			return;

		// disconnect from any associated AP
		if ((mWifiMng.getConnectionInfo().getSupplicantState() == SupplicantState.COMPLETED) ||
			(mWifiMng.getConnectionInfo().getSupplicantState() == SupplicantState.ASSOCIATING) ||
			(mWifiMng.getConnectionInfo().getSupplicantState() == SupplicantState.FOUR_WAY_HANDSHAKE) ||
			(mWifiMng.getConnectionInfo().getSupplicantState() == SupplicantState.GROUP_HANDSHAKE)) {
			mWifiMng.disconnect();
		}
		
		// save all configured AND enabled network configurations
		mWifiConnectionProfiles = new ArrayList<WifiConfiguration>();
		for (WifiConfiguration conf : mWifiMng.getConfiguredNetworks()) {
			if (conf.status != WifiConfiguration.Status.DISABLED) {
				// Remember the connection profile to add it back later
				mWifiConnectionProfiles.add(conf);
				// Disable the connection:
				//mWifiMng.removeNetwork(conf.networkId);
				Log.alwaysWarn(this.getClass().getName(),"DBG: disable network :" + conf.SSID + "  id:" + conf.networkId);
				mWifiMng.disableNetwork(conf.networkId);
			}
		}
		mWifiMng.saveConfiguration();
		mWifiConnectionDisabled = true;
	}
	
	private void enableWifiConnections() {
		if ((mWifiMng==null) || (!mWifiConnectionDisabled))
			return;
		// re-enable all stored configured network configurations
		if (null != mWifiConnectionProfiles) {
			for (WifiConfiguration conf : mWifiConnectionProfiles) {
				int newNetworkId = mWifiMng.addNetwork(conf);
				Log.alwaysWarn(this.getClass().getName(),"DBG: add network :" + conf.SSID + "  id:" +newNetworkId);
				if (newNetworkId>=0) {
					mWifiMng.enableNetwork(newNetworkId, false);
				}
			}
		}
		mWifiMng.saveConfiguration();
		mWifiConnectionProfiles = null;
		mWifiConnectionDisabled = false;
		// Now try to reassociate to the last active AP
		mWifiMng.reassociate();
	}
}
