//______________________________________________________________________________________
//
//  GpsSensor.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.sensors;

import java.lang.ref.WeakReference;


import android.content.Context;
import android.content.ContextWrapper;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.location.LocationProvider;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import com.polestar.models.ISensorListener;
import com.polestar.helpers.Log;
import com.polestar.models.OSLocMeasurement;
import com.polestar.models.SensorNao;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.TSENSORPOWERMODE;
import com.polestar.naosdk.api.TSENSORTYPE;

/**
 * Manages the Network Location interface for Android
 * <p/>
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 * <td>Author</td>
 * <td>Modification date</td>
 * <td>Tracking Number</td>
 * <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 * <td>jchouki</td>
 * <td>19 07 2011</td>
 * <td>trunk</td>
 * <td>Creation of this class</td>
 * </tr>
 * </table>
 *
 * @author jchouki
 */
public class OSLocSensor extends SensorNao implements LocationListener {
    /**
     * Default minimum refresh period for location updates, in milliseconds
     */
    public static final int DEFAULT_REFRESH_DELAY = 1000;

    private static final String OSLOC_SENSOR_THREAD_NAME = "OSLOC_Thread"; // name of the thread OSLOC

    /**
     * if location update is requested
     */
    boolean isOSLocationRequested = false;

    // variables
    private LocationManager mLocManager;
    private ISensorListener mOutputInterface;
    private int previousStatus = LocationProvider.OUT_OF_SERVICE;
    private OSLocMeasurement mCurrentOSLocation;

    /**
     * minimum time interval between location updates
     */
    private long mMinTimeInterval = 0;

    // messages to perform thread actions in the Handler
    protected static final int MSG_START = 1;
    protected static final int MSG_STOP = 2;

    /**
     * nested static class to define the Handler and its actions
     */
    protected static OSLocCollectorHandler mOSLocCollectorHandler = null;

    protected static class OSLocCollectorHandler extends Handler {
        protected WeakReference<OSLocSensor> osLocSensorRef;

        public OSLocCollectorHandler(OSLocSensor bs) {
            osLocSensorRef = new WeakReference<OSLocSensor>(bs);
        }

        public void handleMessage(Message msg) {
            OSLocSensor bs = osLocSensorRef.get();
            if (msg == null || bs == null)
                return;
            switch (msg.what) {
                case MSG_START:
                    bs.startSensorImpl();
                    break;
                case MSG_STOP:
                    bs.stopSensorImpl();
                    break;
                default:
                    Log.alwaysError(this.getClass().getName(), "Unexpected message!");
                    break;
            }
        }
    } // end of OSLocCollectorHandler definition

    /**
     * GpsSensor constructor : creates an intent filter, get a handle to the NMEA services, etc.
     * This constructor is called when launched by an activity.
     *
     * @param caller : reference to calling activity or service
     */
    public OSLocSensor(ContextWrapper caller, ISensorObserver observer) {
        setSensorObserver(observer);
        mThread = new Thread(this);
        mThread.setName(OSLOC_SENSOR_THREAD_NAME);
        mThread.start();

        // copy the reference to the calling activity or service
        mCtxtWrap = caller;
        // set initial status
        mStatus = STATUS_OFF;
    } // end of GpsSensor


    /**
     * checkLocationProviders
     * try if GPS/Network providers are enabled
     */
    public boolean checkLocationProviders() {
        /** when provider is enabled ??*/
        boolean isOSLocEnabled = false;
        //try if provider is enabled
        //exceptions will be thrown if provider is not permitted.
        try {
            isOSLocEnabled = (mLocManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER) || mLocManager.isProviderEnabled(LocationManager.GPS_PROVIDER));
        } catch (Exception ex) {
            Log.alwaysError(this.getClass().getName(), "Exception in OSLocSensor: checkLocationProviders " +
                    "Enabled  > " + ex.getMessage());
        }

        return isOSLocEnabled;
    }

    public static boolean checkLocationProviders(ContextWrapper ctxtWrap) {

        boolean isOSLocEnabled = false;

        LocationManager locationManager = (LocationManager) ctxtWrap.getSystemService(Context.LOCATION_SERVICE);

        try {
            isOSLocEnabled = (locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER) || locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER));
        } catch (Exception ex) {
            Log.alwaysError("OSLocSensor", "Exception in OSLocSensor: checkLocationProviders " +
                    "Enabled  > " + ex.getMessage());
        }

        return isOSLocEnabled;

    }

    /**
     * checkGPSProviders
     * try if GPS/Network providers are enabled
     */
    public boolean checkGPSProviders() {
        /** when provider is enabled ??*/
        boolean isGPSEnabled = false;
        //try if provider is enabled
        //exceptions will be thrown if provider is not permitted.
        try {
            isGPSEnabled = mLocManager.isProviderEnabled(LocationManager.GPS_PROVIDER);
        } catch (Exception ex) {
            Log.alwaysError(this.getClass().getName(), "Exception in OSLocSensor : checkGPSProviders " +
                    "Enabled  > " + ex.getMessage());
        }

        return isGPSEnabled;
    }


    /**
     * thread run
     */
    public void run() {
        Looper.prepare();
        myLooper = Looper.myLooper();

        // get a reference to the Location manager service
        if (mCtxtWrap != null) {
            mLocManager = (LocationManager) mCtxtWrap.getSystemService(Context.LOCATION_SERVICE);

        }
        // check
        if (mLocManager == null) {
            Log.alwaysError(this.getClass().getName(), "Error : Cannot initialize OS Location Service");
        } else {
            mStatus = STATUS_READY;
            //try if provider is enabled
            checkLocationProviders();
        }

        // Initialize the handler to perform actions in this thread
        mOSLocCollectorHandler = new OSLocCollectorHandler(this);

        //create a Network Measurement object
        mCurrentOSLocation = new OSLocMeasurement();

        //starting sensor
        this.mOrder = ORDER_TURN_OFF;

        //looper
        Looper.loop();
    }

    /**
     * destructor : cleanly stops the service.
     */
    protected void finalize() {
        //switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_SERVICE);
    }

    @Override
    public void start() {
        Log.alwaysWarn(this.getClass().getName(), "Starting OSLoc sensors from native .....");
        switchSensor(SensorNao.ORDER_TURN_ON, SensorNao.ORDER_PROVENANCE_SERVICE);
    }

    @Override
    public boolean isRunning() {
        return (mStatus == STATUS_RUNNING);
    }

    @Override
    public void stop() {
        Log.alwaysWarn(this.getClass().getName(), "Stoping OSLoc sensors from native .....");
        switchSensor(SensorNao.ORDER_TURN_OFF, SensorNao.ORDER_PROVENANCE_SERVICE);
    }

    /* (non-Javadoc)
         * @see com.polestar.Sensor#reset()
         */
    @Override
    public void reset() {
        int initialStatus = mStatus;
        switchSensor(SensorNao.ORDER_TURN_OFF, SensorNao.ORDER_PROVENANCE_SERVICE);
        if (initialStatus == STATUS_RUNNING) {
            switchSensor(SensorNao.ORDER_TURN_ON, SensorNao.ORDER_PROVENANCE_SERVICE);
        }
    }

    @Override
    public void setPowerMode(TSENSORPOWERMODE mode) {
        //Not implemented
    }

    /**
     * minimum time interval between location updates
     */
    public void setMinimumTimeIntervalBetweenLocationUpdates(long minTime) {
        this.mMinTimeInterval = minTime;
        Log.alwaysWarn(this.getClass().getName(), "minimum time interval between location updates : " + this.mMinTimeInterval);
    }


    private boolean startSensorImpl() {
        if ((mStatus == STATUS_READY) && (mLocManager != null)) {
            // request location updates
            try {

                if (isOSLocationRequested == false) {
                    Log.restricted(this.getClass().getName(), "requestLocationUpdates...Network");
                    mLocManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, this.mMinTimeInterval, 0, this);
                    Log.restricted(this.getClass().getName(), "requestLocationUpdates...GPS");
                    mLocManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, this.mMinTimeInterval, 0, this);
                    isOSLocationRequested = true;
                }

                mStatus = STATUS_RUNNING;
                this.mOrder = ORDER_TURN_ON;
            } catch (SecurityException se) {
                Log.alwaysError(this.getClass().getName(), "SecurityException in startSensor() > " + se.getMessage());
            } catch (IllegalArgumentException ie) {
                Log.alwaysError(this.getClass().getName(), "IllegalArgumentException in startSensor() > " + ie.getMessage());
            } catch (RuntimeException e) {
                Log.alwaysError(this.getClass().getName(), "RuntimeException in startSensor(): " + e.getMessage());
            }
        } else {
            Log.alwaysWarn(this.getClass().getName(), "Cannot start Network Listener because previous status = " + String.valueOf(mStatus));
        }

        return (mStatus == STATUS_RUNNING);
    }

    /* (non-Javadoc)
     * @see com.polestar.Sensor#start()
     */
    @Override
    public boolean startSensor() {
        boolean returnValue = true;

        // If called from another thread: post execution to this thread
        if (mOSLocCollectorHandler != null) {
            mOSLocCollectorHandler.sendEmptyMessage(MSG_START);
        } else {
            // Error: Handler not initialized
            returnValue = false;
        }

        return returnValue;
    }

    /**
     * Stops the OSLoc sensor. Can be called from any thread.<br>
     */
    @Override
    public void stopSensor() {
        // If called from another thread: post execution to this thread
        if (mOSLocCollectorHandler != null) {
            mOSLocCollectorHandler.sendEmptyMessage(MSG_STOP);
        }
    }

    private void stopSensorImpl() {
        if ((mStatus == STATUS_RUNNING) && (mLocManager != null)) {
            mLocManager.removeUpdates(this);
            isOSLocationRequested = false;
            mStatus = STATUS_READY;
            this.mOrder = ORDER_TURN_OFF;
        }
    }

    /**
     * Register an interface for output GPS measurements
     *
     * @param listener : reference to an object implementing an ISensorListener interface
     */
    public void registerOutputInterface(ISensorListener listener) {
        mOutputInterface = listener;
    } // end of registerGps()

    @Override
    public boolean hasFix() {
        return (mCurrentOSLocation.getLocation().hasAccuracy());
    }

    @Override
    public boolean isOn() {
        return (mStatus == STATUS_RUNNING);
    }

    /**
     * get if sensor is active
     */
    @Override
    public boolean isActive() {
        return this.checkLocationProviders();
    }

    /**
     * get if sensor is HW compatible
     */
    @Override
    public boolean isHardwareCompatible() {
        return (mCtxtWrap.getSystemService(Context.LOCATION_SERVICE) != null);
    }

    /**
     * get if gps sensor is active
     */
    public boolean isGPSActive() {
        return this.checkGPSProviders();
    }

    /**
     * OS location listener.
     */
    public void onLocationChanged(Location location) {

        if (location.hasAccuracy()) {
            notifyStatusUpdate(LocationProvider.AVAILABLE);
        }

        // copy measurement
        mCurrentOSLocation.setLocation(System.currentTimeMillis(), location);

        Log.restricted(this.getClass().getName(), "OS location update; accuracy : " + location.getAccuracy() + " meters");
        //--- end log additional data ---

        if (mOutputInterface
                != null) {
            // notify interface
            mOutputInterface.onNewMeasurement(mCurrentOSLocation);
        }

        if (mSensorObserver != null) {
            mSensorObserver.notifyOfNewData(TSENSORTYPE.LOCOS, mCurrentOSLocation.toByteArray());
        }
    }

    public void onProviderDisabled(String provider) {
        Log.alwaysWarn(this.getClass().toString(), "[" + provider + "] Provider is disabled");
    }

    public void onProviderEnabled(String provider) {
        Log.alwaysWarn(this.getClass().toString(), "[" + provider + "] Provider is enabled");
    }

    public void onStatusChanged(String provider, int status, Bundle extras) {
        notifyStatusUpdate(status);
    }

    /**
     * notify when Status is Updated
     *
     * @param status
     */
    public void notifyStatusUpdate(int status) {
        if (status != previousStatus) {
            // set Loc OS measurement status to 1  at first fix.
            switch (status) {
                case LocationProvider.AVAILABLE:
                    if (mOutputInterface != null) {
                        mOutputInterface.onNewLocOsStatus(true);
                    }
                    break;
                case LocationProvider.OUT_OF_SERVICE: // fall through
                case LocationProvider.TEMPORARILY_UNAVAILABLE:
                default:
                    if (mOutputInterface != null) {
                        mOutputInterface.onNewLocOsStatus(false);
                    }
                    break;
            }
            previousStatus = status;
        } // else ignore redundant notification
    }


    @Override
    public boolean setStateAsBefore() {
        // TODO Auto-generated method stub
        return false;
    }

}